#include "Shaders.h"

#include <GL/glew.h>
#include <memory>
#include <stdexcept>
#include <QDir>
#include <QFile>
#include <QTextStream>

std::string GetShaderLog(unsigned int shader)
{
  int logLength;
  glGetShaderiv(shader, GL_INFO_LOG_LENGTH, &logLength);

  if (logLength <= 1) return "(empty log)";

  std::unique_ptr<char[]> log(new char[logLength]);
  glGetShaderInfoLog(shader, logLength, nullptr, log.get());

  return std::string(log.get());
}

unsigned int CompileShader(unsigned int type, std::string source)
{
  unsigned int shader = glCreateShader(type);

  const char *cStr = source.c_str();
  glShaderSource(shader, 1, &cStr, nullptr);

  glCompileShader(shader);

  int success;
  glGetShaderiv(shader, GL_COMPILE_STATUS, &success);

  if (!success)
    throw std::runtime_error("Compile error\n" + GetShaderLog(shader));

  return shader;
}

std::string GetProgramLog(unsigned int program)
{
  int logLength;
  glGetProgramiv(program, GL_INFO_LOG_LENGTH, &logLength);

  if (logLength <= 1) return "(empty log)";

  std::unique_ptr<char[]> log(new char[logLength]);
  glGetProgramInfoLog(program, logLength, nullptr, log.get());

  return std::string(log.get());
}

unsigned int LinkProgram(unsigned int vertexShader, unsigned int fragmentShader)
{
  unsigned int program = glCreateProgram();

  glAttachShader(program, vertexShader);
  glAttachShader(program, fragmentShader);

  glLinkProgram(program);

  int success;
  glGetProgramiv(program, GL_LINK_STATUS, &success);

  if (!success)
    throw std::runtime_error("Link error\n" + GetProgramLog(program));

  glDetachShader(program, vertexShader);
  glDetachShader(program, fragmentShader);

  return program;
}

ShaderProgram CompileProgramFromFiles(std::string vertFileName, std::string fragFileName)
{
  QDir appDir = QDir::currentPath();
  appDir.cd("Shaders");

  QFile
    vertFile(appDir.filePath(QString::fromStdString(vertFileName))),
    fragFile(appDir.filePath(QString::fromStdString(fragFileName)));

  if (!vertFile.open(QFile::ReadOnly | QFile::Text))
    throw std::runtime_error("Failed to load vertex shader source: " + vertFileName);

  if (!fragFile.open(QFile::ReadOnly | QFile::Text))
    throw std::runtime_error("Failed to load fragment shader source: " + fragFileName);

  QTextStream vertStream(&vertFile);
  QTextStream fragStream(&fragFile);

  QString vertSource = vertStream.readAll();
  QString fragSource = fragStream.readAll();

  ShaderProgram sp;
  sp.VertexShader = CompileShader(GL_VERTEX_SHADER, vertSource.toStdString());
  sp.FragmentShader = CompileShader(GL_FRAGMENT_SHADER, fragSource.toStdString());
  sp.Program = LinkProgram(sp.VertexShader, sp.FragmentShader);
  return sp;
}
