#include "ShowInfoHandler.h"

#include "PointSearch.h"
#include "skelrender.h"
#include "SkeletonController.h"
#include "CutSetRenderer.h"

#include <sstream>


ShowInfoHandler::ShowInfoHandler(SkeletonController &skeletonController)
:
  r_skeletonController(skeletonController),
  up_cutSetRenderer(new CutSetRenderer(r_skeletonController)),
  d_sp_selected(false),
  d_sp_idx(0),
  d_enableCut(false),
  d_enableCutSet(false),
  d_enableFeaturePoints(true),
  d_enableArrow(true),
  d_enableStreamlines(false)
{

}

ShowInfoHandler::~ShowInfoHandler()
{

}

void ShowInfoHandler::onPick(double x, double y, double z, bool picked_something)
{
  std::unique_lock<std::mutex> lck (d_mtx);
  if (picked_something)
  {
    Point3d pick(x, y, z);
    auto d_sm_sp_idx = r_skeletonController.GetSkeletonModel()->getSkelPointSearch().closest(pick);
    auto sm_pick = r_skeletonController.GetSkeletonModel()->GetSkelPoints().at(d_sm_sp_idx);
    d_sp_idx = r_skeletonController.GetFeaturePoints().skelSearch->closest(sm_pick);
    d_selectedPoint = r_skeletonController.GetFeaturePoints().fullSkeleton[d_sp_idx];
    d_sp_selected = true;
  }
  else
  {
    d_sp_selected = false;
  }
  //raise signal that stuff changed
  emit pickChanged();
}

void ShowInfoHandler::draw(Renderer &ren)
{
  std::unique_lock<std::mutex> lck (d_mtx);

  if (d_enableCutSet)
    up_cutSetRenderer->draw();

  if (!d_sp_selected)
    return;

  if (d_enableFeaturePoints)
    ren.drawFeaturePoints(d_sp_idx);
  if (d_enableCut)
    drawCut(ren);
  if (d_enableArrow)
  {
    ren.drawArrow(d_selectedPoint, 10.0f);
    drawEdtArrow(ren);
  }
  if (d_enableStreamlines)
  {
    auto& saliencyEngine = r_skeletonController.GetSaliencyMetricEngine();
    auto& velocity = r_skeletonController.GetSkel().getV();
    auto& graph = r_skeletonController.GetGraph();
    auto& featurePoints = r_skeletonController.GetFeaturePoints();
    auto& map = featurePoints.GetOrConstructCoord2SkelPointIndex();
    float impThreshold = r_skeletonController.GetImpThreshold();

    std::vector<coord3s> streamline;
    saliencyEngine.TraceVelocityStreamLine(nullptr, d_sp_idx, velocity, graph, impThreshold, map, &streamline);
    ren.drawStreamline(streamline);
  }

 
}

void ShowInfoHandler::drawEdtArrow(Renderer& ren)
{
  auto& p = d_selectedPoint;
  Vector velocity = r_skeletonController.GetSkel().getV()(p.x, p.y, p.z);
  coord3s origin(p.x, p.y, p.z);
  coord3s target = r_skeletonController.GetSaliencyMetricEngine().
    FindVelocityCoord(origin, r_skeletonController.GetImpThreshold(), velocity);
  coord3s diff = target - origin;
  Vector arrow(diff.x, diff.y, diff.z);
  ren.drawArrow(p, arrow, 5.0f);
}

void ShowInfoHandler::drawCut(Renderer &ren)
{
  if (! r_skeletonController.getCutController().hasCutSet()) return;
  auto &cuts = r_skeletonController.getCutController().getCutSet().getCuts();
  auto cut_it = cuts.find(d_sp_idx);
  //does our selected point have an associated cut?
  if (cut_it == cuts.end()) return;

  std::vector<Point3d> points;
  auto &graph = r_skeletonController.GetGraph();
  auto &path = *(cut_it->second->up_path);
  for (auto p_node_it = path.begin(); p_node_it != path.end(); ++p_node_it)
  {
    auto n = graph.boundary->at((*p_node_it)->id);
    n.value = 0.5;
    points.push_back(std::move(n));
  }
  ren.draw_cubes(points);
}

float ShowInfoHandler::geoDistbetweenFeaturePoints(size_t index)
{
  auto &graph = r_skeletonController.GetGraph();
  surface::AStar aStarEngine(graph);
  auto& features = r_skeletonController.GetFeaturePoints().featurePoints[index];

  return aStarEngine.shortestPath(features.first, features.second)->length;
}

float ShowInfoHandler::euclDistbetweenFeaturePoints(size_t index)
{
  auto& features = r_skeletonController.GetFeaturePoints().featurePoints[index];
  auto &graph = r_skeletonController.GetGraph();
  const Point3d& pos1 = graph.nodePosition(features.first);
  const Point3d& pos2 = graph.nodePosition(features.second);

  return pos1.dist(pos2);
}

float ShowInfoHandler::getImportance(size_t index)
{
  Point3d& p = r_skeletonController.GetFeaturePoints().fullSkeleton[index];
  auto& importance = r_skeletonController.GetSkel().getImportance();
  return importance(round(p.x), round(p.y), round(p.z));
}

float ShowInfoHandler::getEDT(size_t index)
{
  Point3d& p = r_skeletonController.GetFeaturePoints().fullSkeleton[index];
  auto& EDT = r_skeletonController.GetSkel().getEDT();
  return EDT(round(p.x), round(p.y), round(p.z));
}

float ShowInfoHandler::getSaliency(size_t index)
{
  Point3d& p = r_skeletonController.GetFeaturePoints().fullSkeleton[index];
  auto saliency = r_skeletonController.GetSaliencyVolume();
  if (saliency)
    return saliency->GetInnerVolume()(round(p.x), round(p.y), round(p.z));
  else
    return 0.0f;
}

std::string ShowInfoHandler::getInfoString()
{
  std::unique_lock<std::mutex> lck (d_mtx);
  if (!d_sp_selected) return "";

  auto& saliencyMetricEngine = r_skeletonController.GetSaliencyMetricEngine();
  auto& graph = r_skeletonController.GetGraph();
  auto& velocity = r_skeletonController.GetSkel().getV();
  float imp_t = r_skeletonController.GetImpThreshold();

  float geoDist = geoDistbetweenFeaturePoints(d_sp_idx);
  float euclDist = euclDistbetweenFeaturePoints(d_sp_idx);
  float importance = getImportance(d_sp_idx);
  float edt = getEDT(d_sp_idx);
  float derrivative = saliencyMetricEngine.GetSingleDerrivative(d_sp_idx, graph, velocity, imp_t, 1.0f);
  float saliency = getSaliency(d_sp_idx);

  std::ostringstream info_ss;
  info_ss << "selected:" << d_sp_idx << std::endl;
  info_ss << "importance:" << importance << std::endl;
  info_ss << "geo distance:" << geoDist << std::endl;
  info_ss << "euclidean distance:" << euclDist << std::endl;
  info_ss << "EDT: " << edt << std::endl;
  info_ss << "Derrivative: " << derrivative << std::endl;
  info_ss << "Saliency: " << saliency << std::endl;

  // Return immediately if we have no cut set.
  if(!r_skeletonController.getCutController().hasCutSet())
    return info_ss.str();

  auto &cuts = r_skeletonController.getCutController().getCutSet().getCuts();
  auto cut_it = cuts.find(d_sp_idx);
  // Does our selected point have an associated cut?
  if (cut_it == cuts.end())
  {
    info_ss << "cut: no" <<  std::endl;
    info_ss << "cut length: --" <<  std::endl;
  } else {
    info_ss << "cut: yes" <<  std::endl;
    info_ss << "cut length:" << cut_it->second->up_path->length  <<  std::endl;
  }
  return info_ss.str();
}

void ShowInfoHandler::setEnableCut(bool value)
{
  if (value != d_enableCut)
  {
    d_enableCut = value;
    emit enableCutChanged(value);
  }
}

void ShowInfoHandler::setEnableCutSet(bool value)
{
  if (value != d_enableCutSet)
  {
    d_enableCutSet = value;
    emit enableCutSetChanged(value);
  }
}

void ShowInfoHandler::setEnableFeaturePoints(bool value)
{
  if (value != d_enableFeaturePoints)
  {
    d_enableFeaturePoints = value;
    emit enableFeaturePointsChanged(value);
  }
}

void ShowInfoHandler::setEnableStreamlines(bool value)
{
  d_enableStreamlines = value;
}
