#include "glwrapper.h"

#include "skelrender.h"
#include "collapseSkel3d.h"
#include "utils.h"
#include "Point2d.h"
#include "FeaturePoints.h"
#include "ImportanceVolume.h"

#ifdef WIN32
#define _USE_MATH_DEFINES
#endif
#include <math.h>
#include <string.h>
#include <string>
#include <iostream>
#include <boost/smart_ptr/shared_ptr.hpp>

using namespace std;


extern float   zprScale;                      //From the zpr library
extern double*
minv;                        //Inverse of modelview matrix, from zpr library

static vector<coord3s> xx; //!!

void glutDrawString(const char* s)
{
  while (*s) glutBitmapCharacter(GLUT_BITMAP_HELVETICA_12, *s++);
}




char *textFileRead(char* buffer, const string &fn)                      //
{
  FILE* fp = fopen(fn.c_str(), "rt");
  if (!fp) return 0;

  fseek(fp, 0, SEEK_END);
  int count = ftell(fp);
  rewind(fp);
  if (count <= 0) return 0;

  count = fread(buffer, sizeof(char), count, fp);
  buffer[count] = '\0';
  fclose(fp);

  return buffer;
}


static void printShaderLog(GLuint obj)
{
  GLint infoLogLength = 0;
  GLsizei charsWritten  = 0;
  GLchar *infoLog;

  glGetShaderiv(obj, GL_INFO_LOG_LENGTH, &infoLogLength);

  if (infoLogLength > 1)
  {
    infoLog = (char *) malloc(infoLogLength);
    glGetShaderInfoLog(obj, infoLogLength, &charsWritten, infoLog);
    printf("\n%s\n", infoLog);
    free(infoLog);
  }
}

static void printProgramLog(GLuint obj)
{
  GLint infoLogLength = 0;
  GLsizei charsWritten  = 0;
  GLchar *infoLog;

  glGetProgramiv(obj, GL_INFO_LOG_LENGTH, &infoLogLength);

  if (infoLogLength > 1)
  {
    infoLog = (char *) malloc(infoLogLength);
    glGetProgramInfoLog(obj, infoLogLength, &charsWritten, infoLog);
    printf("\n%s\n", infoLog);
    free(infoLog);
  }
}

static void checkLink(GLuint prog)
{
  GLint stat;
  glGetProgramiv(prog, GL_LINK_STATUS, &stat);
  if (!stat)
  {
    GLchar log[1000];
    GLsizei len;
    glGetProgramInfoLog(prog, 1000, &len, log);
    fprintf(stderr, "Linker error:\n%s\n", log);
  }
}

float imp_lowest = 0;


Renderer::Renderer(collapseSkel3d &skel_, float imp_thr_)
:
  imp_thr(imp_thr_),
  skel(skel_),
  featurePoints(nullptr)
{
  //!!
  imp_lowest = imp_thr;

  quad = 0;
  sphere_faces   = 8;        //resolution of the GLUT spheres (small=fast)
  light_specular = 0.7;      //specular amount
  light_diffuse  = 0.7;      //diffuse amount
  specular_power = 10;       //shininess
  use_mipmaps    = true;     //use texture mipmaps for splatting or not
  ball_mode      =
    BALL_SPRITE;  //use shaders for ball splatting; best/fastest results
  use_floating_textures =
    true;  //whether to use floating-point or 24-bit fixed-point textures for the splat height
  light_pos[0] = 0;        //light direction
  light_pos[1] = 0;
  light_pos[2] = 1;
  light_pos[3] = 0;
  vis_imp_min  = 0;
  vis_imp_max  = 1;
  smooth_shading = true;     //use vertex normals (that is, if we have them)
  imp_alpha_min = 1.0;       //
  imp_alpha_slope = 1;
  ball_dt_offset = 0.5;
  splat_dt_offset = 0.5;
  splat_size =
    256;        //size of largest texture splat; 256 seems more than enough
  ball_size =
    2.0;         //ball radius for constant-radius skeleton drawing. Bit larger than a voxel's size
  surf_alpha = 0.3f;

  glewExperimental = true;
  GLenum err = glewInit();
  if (err != GLEW_OK)
  {
    //Problem: glewInit failed, something is seriously wrong.
    cout << "glewInit failed, aborting." << endl;
  }

  if (!(GLEW_ARB_fragment_program &&
      GLEW_ARB_vertex_program &&
      GLEW_EXT_geometry_shader4))
  {
    cerr << "Error: Unable to load required GL extensions, exiting" << endl;
    exit(-1);
  }

  glGenTextures(2,
                balltex);               //Generate a texture-ID for the ball splat

  GLuint geom_objs = glGenLists(
                       1);              //Make some display lists for the stuff we'll draw often
  splat_quad = geom_objs;                    //First list: splat quad
  glNewList(splat_quad,
            GL_COMPILE);             //This draws a [-1,1]-sized, textured, quad
  glBegin(GL_QUADS);
  glTexCoord2f(0, 0);
  glVertex2d(-1, -1);
  glTexCoord2f(1, 0);
  glVertex2d( 1, -1);
  glTexCoord2f(1, 1);
  glVertex2d( 1, 1);
  glTexCoord2f(0, 1);
  glVertex2d(-1, 1);
  glEnd();
  glEndList();

  char buffer[10000];
  glGenProgramsARB(5, ball_shader);              //Create 5 shaders:

  glBindProgramARB(GL_FRAGMENT_PROGRAM_ARB,
                   ball_shader[0]);       //1. ARB shader for quad-based splatting, RGBA textures
  char* rgba_s = textFileRead(buffer, "Shaders/ball_shader_rgba.arb");
  glProgramStringARB(GL_FRAGMENT_PROGRAM_ARB, GL_PROGRAM_FORMAT_ASCII_ARB,
                     strlen(rgba_s), rgba_s);
  glBindProgramARB(GL_FRAGMENT_PROGRAM_ARB,
                   ball_shader[1]);       //2. ARB shader for quad-based splatting, float textures
  char* float_s = textFileRead(buffer, "Shaders/ball_shader_float.arb");
  glProgramStringARB(GL_FRAGMENT_PROGRAM_ARB, GL_PROGRAM_FORMAT_ASCII_ARB,
                     strlen(float_s), float_s);

  ball_shader[2] = glCreateShader(
                     GL_VERTEX_SHADER);         //3. Vertex shader, sprite-based splatting
  char* vs = textFileRead(buffer, "Shaders/ball_shader.vert");
  glShaderSource(ball_shader[2], 1, (const char**)&vs, 0);
  glCompileShader(ball_shader[2]);
  printShaderLog(ball_shader[2]);

  ball_shader[3] = glCreateShader(
                     GL_FRAGMENT_SHADER);         //4. Fragment shader, sprite-based splatting
  char *fs = textFileRead(buffer, "Shaders/ball_shader.frag");
  glShaderSource(ball_shader[3], 1, (const char**)&fs, 0);
  glCompileShader(ball_shader[3]);
  printShaderLog(ball_shader[3]);

  ball_shader[4] = glCreateShader(
                     GL_GEOMETRY_SHADER_ARB);       //5. Geometry shader, sprite-based splatting
  char *gs = textFileRead(buffer, "Shaders/ball_shader.geom");
  glShaderSource(ball_shader[4], 1, (const char**)&gs, 0);
  glCompileShader(ball_shader[4]);
  printShaderLog(ball_shader[4]);

  ball_prog =
    glCreateProgram();             //Assemble last 3 shaders in a 'program'
  glAttachShader(ball_prog, ball_shader[2]);
  glAttachShader(ball_prog, ball_shader[3]);
  glAttachShader(ball_prog, ball_shader[4]);
  glProgramParameteriARB(ball_prog, GL_GEOMETRY_INPUT_TYPE_ARB, GL_POINTS);
  glProgramParameteriARB(ball_prog, GL_GEOMETRY_OUTPUT_TYPE_ARB,
                         GL_TRIANGLE_STRIP);
  int temp = 4;
  glProgramParameteriARB(ball_prog, GL_GEOMETRY_VERTICES_OUT_ARB, temp);
  glLinkProgram(ball_prog);
  printProgramLog(ball_prog);
  checkLink(ball_prog);

  //glUseProgram(0);
}



void Renderer::initGL()
{
  float lDiff[4], lSpec[4], lAmb[4];
  lAmb[0]  = lAmb[1]  = lAmb[2]  = 0;
  lAmb[3] = 0;
  lDiff[0] = lDiff[1] = lDiff[2] = light_diffuse;
  lDiff[3] = 1;
  lSpec[0] = lSpec[1] = lSpec[2] = lSpec[3] = light_specular;

  glMaterialfv(GL_FRONT_AND_BACK, GL_AMBIENT, lAmb);
  glMaterialfv(GL_FRONT_AND_BACK, GL_DIFFUSE, lDiff);
  glMaterialfv(GL_FRONT_AND_BACK, GL_SPECULAR, lSpec);
  glMaterialf(GL_FRONT_AND_BACK, GL_SHININESS, specular_power);

  GLfloat white[] = {1, 1, 1, 1};
  glLightfv(GL_LIGHT0, GL_POSITION, light_pos);
  glLightfv(GL_LIGHT0, GL_DIFFUSE, white);
  glLightfv(GL_LIGHT0, GL_SPECULAR, white);
  glEnable(GL_NORMALIZE);

  glEnable(GL_LIGHTING);
  glEnable(GL_LIGHT0);

  glEnable(GL_DEPTH_TEST);
  glEnable(GL_PROGRAM_POINT_SIZE);

  glPixelStorei(GL_PACK_ALIGNMENT, 1);
  glPixelStorei(GL_UNPACK_ALIGNMENT, 1);

  quad = gluNewQuadric();
  makeTextures();
}


Renderer::~Renderer()
{
}

void Renderer::createSplatFloat(int SZ, GLfloat* img,
                                int level) //Creates a single SZ*SZ RGBA texture splat (floating-point)
{
  const float half = 0.5;
  const float C = SZ / 2.0;
  const float C2 = C *
                   C;                  //Generate a half-sphere height profile, encode it as a luminance-alpha texture
  for (int i = 0, idx = 0; i < SZ;
       ++i)        //The height should be normalized in [0..1]; alpha indicates the valid vs nonvalid pixels
    for (int j = 0; j < SZ; ++j, idx += 4)
    {
      float x  = i - C + half, y = j - C + half;
      int   r2 = x * x + y * y;
      if (r2 > C2)                     //Point outside the ball: color irrelevant, and it's transparent
      {
        img[idx] = img[idx + 1] = img[idx + 2] = img[idx + 3] = 0;
      }
      else                           //Point in the ball: store a RGBA texel with:
      {
        // - R:  ball height [0..1]
        float   D = sqrt(C2 - r2);              // - G:  ball shading [0..1]
        float dfx = -x / D, dfy = -y / D;         // - A:  transparency mask (1)
        Point3d n(dfx, dfy,
                  1);               //  -B:  ball height, high-precision part [0..1/255]
        n.normalize();                    //This is the ball's normal vector

        float dp   = fabs(n.dot(
                            light_pos));          //Simple Phong lighting model using same params as GLUT balls
        float I    = light_diffuse * dp + light_specular * pow(dp, specular_power);
        if (I > 1) I = 1;
        img[idx]   = D / C;                 //floating-point height
        img[idx + 1] = 0;
        img[idx + 2] = 0;
        img[idx + 3] = I;                 //intensity
      }
    }

  glTexImage2D(GL_TEXTURE_2D, level, GL_LUMINANCE_ALPHA32F_ARB, SZ, SZ, 0,
               GL_RGBA, GL_FLOAT, img);
  //We use a luminance-alpha internal format, since it's apparently cheaper and faster than RGBA
}





void Renderer::createSplatRGBA(int SZ, GLubyte* img,
                               int level) //Creates a single SZ*SZ RGBA texture splat (8-bit per channel)
{
  const float half = 0.5;
  const float C = SZ / 2.0;
  const float C2 = C *
                   C;                   //Generate a half-sphere height profile, encode it as a luminance-alpha texture
  for (int i = 0, idx = 0; i < SZ;
       ++i)         //The height should be normalized in [0..1]; alpha indicates the valid vs nonvalid pixels
    for (int j = 0; j < SZ; ++j, idx += 4)
    {
      float x  = i - C + half, y = j - C + half;
      int   r2 = x * x + y * y;
      if (r2 > C2)                      //Point outside the ball: color irrelevant, and it's transparent
      {
        img[idx] = img[idx + 1] = img[idx + 2] = img[idx + 3] = 0;
      }
      else                          //Point in the ball: store a RGBA texel with:
      {
        // - R:  ball height [0..1]
        float   D = sqrt(C2 - r2);            // - G:  ball shading [0..1]
        float dfx = -x / D, dfy = -y / D;       // - A:  transparency mask (1)
        Point3d n(dfx, dfy,
                  1);             //  -B:  ball height, high-precision part [0..1/255]
        n.normalize();                    //This is the ball's normal vector

        float dp   = fabs(n.dot(
                            light_pos));        //Simple Phong lighting model using same params as the OpenGL model
        float I    = light_diffuse * dp + light_specular * pow(dp,
                     specular_power); //!!!May not be fully correct, I recall I had to use here the view vector for the specular term
        if (I > 1) I = 1;

        unsigned int v = 255.0 * 255 * 255 * (D /
                                              C); //24-bit fixed-point height, normalized on [0..255^3-1]

        img[idx + 2] = v % 256;
        v >>= 8;          //bits 23-16 of fixed-point height
        img[idx + 1] = v % 256;
        v >>= 8;          //bits 15-8 of fixed-point height
        img[idx  ] = v % 256;               //bits 7-0 of fixed-point height
        img[idx + 3] = 255 * I;             //intensity
      }
    }


  glTexImage2D(GL_TEXTURE_2D, level, GL_RGBA, SZ, SZ, 0, GL_RGBA,
               GL_UNSIGNED_BYTE, img);
}

int Renderer::getScalingMode() const
{
  if (scaling_mode == -1) return (use_mipmaps) ? GL_LINEAR_MIPMAP_LINEAR :
                                   GL_LINEAR;
  return scaling_mode;
}


void Renderer::makeTextures(int
                            sm)               //Generate splatting textures and other related tools
{
  scaling_mode =
    sm;                     //Remember if we specified a scaling mode or using 'defaults'

  GLfloat* img = new
  GLfloat[4 * splat_size * splat_size]; //Allocate some buffer where we'll synthesize the texture images
  GLubyte* imgb = (GLubyte*)img;

  glBindTexture(GL_TEXTURE_2D,
                balltex[0]);            //Create mipmap levels by hand (to be accurate) for both the fixed-point
  for (int l = 0, SZ = splat_size; SZ >= 1;
       SZ = SZ / 2,
       ++l) //and floating-point textures. We'll use whichever we want later.
    createSplatRGBA(SZ, imgb, l);

  glBindTexture(GL_TEXTURE_2D,
                balltex[1]);            //Create mipmap floating-point textures
  for (int l = 0, SZ = splat_size; SZ >= 1; SZ = SZ / 2, ++l) //
    createSplatFloat(SZ, img, l);

  delete[] img;
}

void Renderer::drawFixedBalls()               //Draw fixed-radius skeleton balls for skeleton-rendering
{
  draw_balls(skeletonModel->GetSkelPoints(), true, true, &skel.getImportance());
}

void Renderer::drawBalls()                    //Draw skeleton balls for surface reconstruction
{
  draw_balls(skeletonModel->GetSkelPoints(), false, false, &skel.getImportance());
}

void Renderer::drawThinBalls()
{
  draw_balls(skeletonModel->GetThinPoints(), false, true, &skel.getImportance());
}

void Renderer::drawBoundarySplats()
{

}

void Renderer::drawBoundaryFixedBalls()
{
}

void Renderer::drawFeaturePoints(int skelpoint) 
{
  vector<Point3d> skp(1);             //1. Draw the skeleton point
  skp[0] = featurePoints->fullSkeleton[skelpoint];
  draw_balls(skp, false, true, &skel.getImportance());
  auto& input_points = inputModel->GetThinPoints();

  const FeaturePoints::ExtFeatures &fPoints =
    featurePoints->extFeaturePoints[skelpoint];

  float dmax = 0, dmin = 1.0e6;
  vector<Point3d> points(fPoints.size());
  int i = 0;
  for (FeaturePoints::ExtFeatures::const_iterator it = fPoints.begin();
       it != fPoints.end(); ++it, ++i)
  {
    points[i] = input_points[*it];
    points[i].value = points[i].dist(skp[0]);
    dmin = min(dmin, points[i].value);
    dmax = max(dmax, points[i].value);
  }

  float range = dmax - dmin;
  if (range < 1.0e-6) range = 1;
  for (size_t i = 0; i < fPoints.size(); ++i)
    points[i].value = (points[i].value - dmin) / range;

  draw_balls(points, true, true);
}

void Renderer::drawStreamline(vector<coord3s>& points)
{
  vector<Point3d> line;

  for (coord3s& p : points)
    line.push_back(Point3d(p.x, p.y, p.z, imp_max));
  
  draw_balls(line, true, true);
}

void Renderer::drawProjectedSkelBoundary()
{
  //ALEX: Far from ready...
  /*
  int BP = boundary_points.size();
  vector<Point3d> pb(BP);

  for(int i=0;i<BP;++i)
  {
    int bp = sbp[i];
    const Point3d& p = init_points[featurePoints->closestFeaturePoints[bp]];  //!!problem: map bp to full-skel-idx
    pb[i] = p;
  }

  draw_balls(pb,true,true);
  */
}

void Renderer::SetUniformBallScale(float scale)
{
  GLint location = glGetUniformLocation(ball_prog, "ballMaxScale");
  glProgramUniform1f(ball_prog, location, scale);
}


void Renderer::draw_balls(const vector<Point3d> &points, bool use_colormap,
                          bool use_constant_radius, const Volume<float>* scalars)
{
  //Draw skeleton balls for surface reconstruction
  static float r_max = 0.0;
  double mv[16], pr[16], NDC_factor(0.0);
  int splat_mode;
  float maxBallRadiusScale = 1.0f;

  glShadeModel((smooth_shading &&
                (ball_mode == BALL_GEOM)) ? GL_SMOOTH :
               GL_FLAT); //In splatting mode, we don't need smooth shading (actually, we don't need shading at all)

  if (ball_mode != BALL_GEOM)                 //BALL_QUAD or BALL_SPRITE modes:
  {
    // -we don't need lighting, we do it ourselves in the textures
    glDisable(GL_LIGHTING);                   // -we need texturing of course
    glEnable(GL_TEXTURE_2D);                  // -we don't need blending, since we use the alpha test
    glDisable(GL_BLEND);                    // -we need a fragment program
    glEnable(GL_ALPHA_TEST);                  //Needed to make transparency work
    //!!Normally, 0 should be enough here; however, it creates some ugly thin-black border effects with mipmapping
    glAlphaFunc(GL_GREATER, 0.1);
    //See if we use fixed-point or floating-point balls.
    splat_mode = (use_floating_textures) ? 1 : 0;
    //The two have different textures and shaders
    glBindTexture(GL_TEXTURE_2D, balltex[splat_mode]);
    glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_REPLACE);
    glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP);
    glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP);
    glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, getScalingMode());
    glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
  }

  //Render balls as GL_QUAD textured billboards:
  if (ball_mode == BALL_QUAD)
  {
    glEnable(GL_DEPTH_TEST);
    
    //Use fragment shader for color+depth computations
    glEnable(GL_FRAGMENT_PROGRAM_ARB);
    glBindProgramARB(GL_FRAGMENT_PROGRAM_ARB, ball_shader[splat_mode]);

    //Get projection matrix to grab near/far plane information
    glGetDoublev(GL_PROJECTION_MATRIX, pr);
    //Save the modelview matrix for later
    glGetDoublev(GL_MODELVIEW_MATRIX, mv);
    //Precompute some factors for the NDC coord transform below
    NDC_factor = 0.5 * pr[14] / pr[11];
    //Disable modelview: we'll do it by hand below
    glMatrixMode(GL_MODELVIEW);
    glPushMatrix();
    glLoadIdentity();
  }
  //Render balls as points. Billboard geometry is created using geom. shaders.
  else if (ball_mode == BALL_SPRITE)
  {
    glUseProgram(ball_prog);

    //Normalization scale for the ball radius, since colors should be in [0, 1]
    maxBallRadiusScale = r_max * zprScale;
    SetUniformBallScale(maxBallRadiusScale);

    //Render points which are converted in quads by geom. shader
    glBegin(GL_POINTS);
  }
  else //ball_mode==BALL_GEOM                   //BALL_GEOM mode: we'll draw spheres
  {
    glEnable(GL_LIGHTING);
  }



  float range     = imp_max - imp_min;
  float vis_range = vis_imp_max - vis_imp_min;

  //Draw all skeleton points:
  for (int i = 0, iend = points.size(); i < iend; ++i)
  {
    const Point3d &s = points[i];
    double r = ((use_constant_radius) ? ball_size : s.value) + ball_dt_offset;

    //The max radius is determined here on the fly; so from the 2nd frame being rendered, it should be correct.
    if (r > r_max) r_max = r;
    float imp = 0;


    if (use_constant_radius)
    {
      if (use_colormap)
      {
        //Where to get the data from: own point-value or 3D scalar volume
        if (scalars)
        {
          //Importance, normalized in [0,1]
          imp = (*scalars)(s.x, s.y, s.z);
          imp = (imp - imp_min) / range;

          //Clamp importance to [vis_imp_min, vis_imp_max] for visualization purposes
          imp = (imp < vis_imp_min) ? 0
                :       
                (imp > vis_imp_max) ? 1
                :       //(the full color spectrum will be used only on this visual range)
                (imp - vis_imp_min) / vis_range;
        }
        else imp = s.value;
      }
    }

    if (ball_mode == BALL_SPRITE)
    {
      //The color passes the splat-radius and colormap-type to shaders
      glColor4f(0.0f, imp, r / maxBallRadiusScale, (float)use_colormap);
      //Scaling affects splat's radius; since geom. shader assumes scaling=1.0f, we prescale radius by zprScaling factor.
      glVertex3fv(s.data);
    }
    else //BALL_GEOM or BALL_QUAD modes:
    {
      glPushMatrix();
      //Draw quad of sizes [2r,2r] as billboard,
      //i.e. parallel to view plane, and centered in 3D at skel point
      if (ball_mode == BALL_QUAD)
      {
        //Following 3 lines compute the 3D world location s transformed by the modelview matrix
        double z = mv[2] * s.x + mv[6] * s.y + mv[10] * s.z + mv[14]; 
        float  x = mv[0] * s.x + mv[4] * s.y +  mv[8] * s.z + mv[12];
        float  y = mv[1] * s.x + mv[5] * s.y +  mv[9] * s.z + mv[13];
        glTranslatef(x, y, z);
        //We don't need to scale in z, this is handled by the texture
        glScalef(r, r, 1);

        //This should be actually r (along the depth axis) in window-coordinates
        float rNDC =  NDC_factor * r / z / (z + r);
        glColor3f(rNDC, 0,
                  0);                //Color passes the splat radius (in NDC) to the pixel shader
        //No point splatting: Draw a textured quad (from a list for some extra speed)
        glCallList(splat_quad);
      }
      else //BALL_GEOM mode
      {
        glTranslatef(s.x, s.y, s.z);
        //If sphere_faces==2, we actually draw a cube
        //(since I don't know how to get this with glutSolidSphere)
        if (sphere_faces == 2) glutSolidCube( r);
        //Classical sphere drawing
        else glutSolidSphere(r, sphere_faces, sphere_faces);
      }
      //Restore modelview matrix (in case we modified it)
      glPopMatrix();
    }
  }

  //Revert to standard settings
  if (ball_mode == BALL_SPRITE)
  {
    glEnd();
    glUseProgram(0);
    glEnable(GL_LIGHTING);
    glDisable(GL_TEXTURE_2D);
    glDisable(GL_ALPHA_TEST);
  }
  else if (ball_mode == BALL_QUAD)
  {
    glDisable(GL_FRAGMENT_PROGRAM_ARB);
    glEnable(GL_LIGHTING);
    glDisable(GL_BLEND);
    glDisable(GL_TEXTURE_2D);
    glDisable(GL_ALPHA_TEST);
    glDisable(GL_DEPTH_TEST);
    glMatrixMode(GL_MODELVIEW);
    glPopMatrix();
  }
}



void Renderer::drawLog()
{
  glMatrixMode(GL_MODELVIEW);
  glPushMatrix();
  glLoadIdentity();
  glMatrixMode(GL_PROJECTION);
  glPushMatrix();
  glLoadIdentity();
  glDisable(GL_DEPTH_TEST);
  glDisable(GL_LIGHTING);
  glEnable(GL_LINE_SMOOTH );
  glHint(GL_LINE_SMOOTH_HINT, GL_NICEST);
  glEnable(GL_BLEND);
  glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);

  const LogData &log = skel.getLog();
  const int   N   = log.size();
  const float eps = 0.01;
  const float ox  = eps - 1, oy = eps - 1;
  const float dx  = 1.0 / N, dy = 0.5;
  /*const int   wx  = glutGet(GLUT_WINDOW_WIDTH);*/
  const int   wy  = glutGet(GLUT_WINDOW_HEIGHT);
  const float txty = 2.0 / wy * 15;

  vector<Point2d> ipv, epv, cdv, tdv;

  int mp = std::max(log.interface_points_max, log.eroded_points_max);

  for (size_t i = 0; i < log.size(); ++i)
  {
    const LogEntry &e = log[i];

    glColor3f(1, 0, 0);

    float ip = float(e.interface_points) / mp;
    float ep = float(e.eroded_points) / mp;
    float cd = float(e.crt_distance) / log.crt_distance_max;
    float td = float(e.tot_density) / log.tot_density_max;

    float x  = ox + i * dx;

    ipv.push_back(Point2d(x, oy + dy * ip));
    epv.push_back(Point2d(x, oy + dy * ep));
    cdv.push_back(Point2d(x, oy + dy * cd));
    tdv.push_back(Point2d(x, oy + dy * td));
  }

  glColor3f(1, 0, 0);
  glRasterPos2f(ox + 0.25, oy + dy);
  glutDrawString("Interface points");
  glLineWidth(4);
  glBegin(GL_LINE_STRIP);
  for (size_t i = 0; i < log.size(); ++i)
    glVertex2f(ipv[i].x, ipv[i].y);
  glEnd();

  glColor3f(0, 1, 0);
  glRasterPos2f(ox + 0.25, oy + dy - txty);
  glutDrawString("Eroded points");
  glLineWidth(3);
  glBegin(GL_LINE_STRIP);
  for (size_t i = 0; i < log.size(); ++i)
    glVertex2f(epv[i].x, epv[i].y);
  glEnd();

  glColor3f(0, 0, 1);
  glRasterPos2f(ox + 0.25, oy + dy - 2 * txty);
  glutDrawString("Total density");
  glLineWidth(2);
  glBegin(GL_LINE_STRIP);
  for (size_t i = 0; i < log.size(); ++i)
    glVertex2f(tdv[i].x, tdv[i].y);
  glEnd();

  glColor3f(0, 1, 1);
  glRasterPos2f(ox + 0.25, oy + dy - 3 * txty);
  glutDrawString("Current distance");
  glLineWidth(1);
  glBegin(GL_LINE_STRIP);
  for (size_t i = 0; i < log.size(); ++i)
    glVertex2f(cdv[i].x, cdv[i].y);
  glEnd();


  glDisable(GL_BLEND);
  glEnable(GL_LIGHTING);
  glEnable(GL_DEPTH_TEST);
  glPopMatrix();
  glMatrixMode(GL_MODELVIEW);
  glPopMatrix();
}

void Renderer::setModels(std::shared_ptr<SkeletonModel> skeletonModel, std::shared_ptr<SkeletonModel> inputModel)
{
  this->skeletonModel = skeletonModel;
  this->inputModel = inputModel;

  if (pointRenderer)
    return;

  const int wx = glutGet(GLUT_WINDOW_WIDTH);
  const int wy = glutGet(GLUT_WINDOW_HEIGHT);

  pointRenderer.reset(new SkeletonPointRenderer(inputModel, skeletonModel, scalars, scalarMin, scalarMax, glm::ivec2(wx, wy) ));
}

void Renderer::setScalar(Volume<float>* scalars, float scalarMin, float scalarMax)
{
  this->scalars = scalars;
  this->scalarMin = scalarMin;
  this->scalarMax = scalarMax;

  pointRenderer->SetScalarVolume(scalars, scalarMin, scalarMax);
}

void Renderer::setScalar(ImportanceVolume & importanceVolume)
{
  setScalar(&importanceVolume.GetInnerVolume(),
    importanceVolume.GetMinImportance(), importanceVolume.GetMaxImportance());

  imp_min = scalarMin;
  imp_max = scalarMax;
  vis_imp_max = scalarMax;
}

void Renderer::setFeaturePoints(FeaturePoints* featurePoints)
{
  this->featurePoints = featurePoints;
}

void Renderer::resizeWindow(int width, int height)
{
  pointRenderer->ResizeCanvas(glm::ivec2(width, height));
}

void Renderer::drawSplats()
{
  //draw_splats(skeletonModel->GetSkelPoints(), skeletonModel->GetSkelNormals(), &skel.getImportance());
  pointRenderer->DrawThickSkeletonPoints();
}

void Renderer::drawInputSplats()
{
  auto& input_points = inputModel->GetThinPoints();
  auto& init_normals = inputModel->GetThinNormals();
  //draw_splats(input_points, init_normals);

  if (reconstructionModel && drawReconstruction)
    pointRenderer->DrawReconstructionModel();
  else
    pointRenderer->DrawInputModel();
}


void Renderer::draw_splats(const vector<Point3d> &points,
                           const vector<Point3d> &normals, const Volume<float>* scalar_data)
{
  if (imp_alpha_min < 1)
  {
    glEnable(GL_BLEND);
    glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
  }
  else
    glDisable(GL_BLEND);

  float col[4];
  float range = imp_max - imp_min;
  float vis_range = vis_imp_max - vis_imp_min;
  float psz = sqrt(3) * zprScale +
              splat_dt_offset; //Estimate splat size to be more or less a voxel-size in screen space
  if (psz < 1) psz = 1;
  glEnable(GL_POINT_SMOOTH);            //length sqrt(3), as function of the viewing direction, zoom factor, etc.
  glHint(GL_POINT_SMOOTH_HINT, GL_NICEST);
  glPointSize(psz);

  float onorm[3];
  float n[3] = {0, 0, 1};
  onorm[0] = minv[0] * n[0] + minv[4] * n[1] + minv[8] * n[2];
  onorm[1] = minv[1] * n[0] + minv[5] * n[1] + minv[9] * n[2];
  onorm[2] = minv[2] * n[0] + minv[6] * n[1] + minv[10] * n[2];

  glBegin(GL_POINTS);
  for (int pid = 0, NP = points.size(); pid < NP; ++pid)
  {
    const Point3d &p = points[pid];

    if (scalar_data)              //Color-map some scalar volume:
    {
      float val = (*scalar_data)(p.x, p.y, p.z);
      val = (val - imp_min) / range;    //Importance, normalized in [0,1]

      val = (val < vis_imp_min) ? 0
            :   //Clamp importance to [vis_imp_min, vis_imp_max] for visualization purposes
            (val > vis_imp_max) ? 1
            :   //(the full color spectrum will be used only on this visual range)
            (val - vis_imp_min) / vis_range;

      float2rgb(val, col);
      col[3] = imp_alpha_min + (1 - imp_alpha_min) * pow(val, imp_alpha_slope);
      //Set color+transparency as function of importance
    }
    else                    //Color-map actual point values:
    {
      float val = p.value;
      float2rgb(val, col);
      col[3] = 1;
    }

    glColor4fv(col);

    Point3d N = normals[pid].data;        //Get cached normal vector of splat
    if (N.x == 0 && N.y == 0 &&
        N.z == 0) N =
            onorm; //If no normal was computed for this point, use view-based one
    else if (N.dot(onorm) <
             0)        //If the caller gave an orientation-hint for the normal,
    {
      N[0] *= -1;  //use it (swap computed normal along hint's orientation)
      N[1] *= -1;
      N[2] *= -1;
    }

    glNormal3fv(N.data);
    glVertex3fv(p.data);
  }

  glEnd();
  glDisable(GL_BLEND);
}


void Renderer::drawThinSplats()
{
  pointRenderer->DrawThinSkeletonPoints();
}





void Renderer::draw_cubes(const vector<Point3d> &points,
                          const Volume<float>* scalar_data)
{
  if (imp_alpha_min < 1)
  {
    glEnable(GL_BLEND);
    glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
  }
  else
    glDisable(GL_BLEND);

  float col[4];
  float range     = imp_max - imp_min;
  float vis_range = vis_imp_max - vis_imp_min;

  for (int pid = 0, NP = points.size(); pid < NP; ++pid)
  {
    const Point3d &p = points[pid];
    if (scalar_data)              //Color-map scalar volume:
    {
      float val = (*scalar_data)(p.x, p.y, p.z);
      val = (val - imp_min) / range;    //Importance, normalized in [0,1]
      val = (val < vis_imp_min) ? 0
            :   //Clamp scalar value to [vis_imp_min, vis_imp_max] for visualization purposes
            (val > vis_imp_max) ? 1
            :   //(the full color spectrum will be used only on this visual range)
            (val - vis_imp_min) / vis_range;

      float2rgb(val, col);
      col[3] = imp_alpha_min + (1 - imp_alpha_min) * pow(val, imp_alpha_slope);
      //Set transparency as function of importance
    }
    else                    //Color-map point values:
    {
      float val = p.value;
      float2rgb(val, col);
      col[3] = 1;
    }

    glPushMatrix();
    glColor3fv(col);
    glTranslatef(p.x, p.y, p.z);
    glutSolidCube(1.0f);
    glPopMatrix();
  }

  glDisable(GL_BLEND);
}


void Renderer::drawThinCubes()
{
  draw_cubes(skeletonModel->GetThinPoints());
}

void Renderer::drawCubes()
{
  draw_cubes(skeletonModel->GetSkelPoints(), &skel.getImportance());
}

void Renderer::drawBoundaryCubes()
{
}

void Renderer::drawInputCubes()
{
  shared_ptr<SkeletonModel> input = drawReconstruction && reconstructionModel
    ? reconstructionModel
    : inputModel;

  auto& input_points = input->GetThinPoints();
  draw_cubes(input_points);
}


void Renderer::drawSurface()
{
  CIsoSurface<byte> &surf = skel.getSurface();
  if (!surf.IsSurfaceValid()) return;

  const float col[4] = {1.0f, 1.0f, 1.0f, surf_alpha};

  glColor4fv(col);

  glEnable(GL_BLEND);
  glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
  glDepthMask((surf_alpha < 1) ? GL_FALSE : GL_TRUE);

  glBegin(GL_TRIANGLES);
  for (size_t i = 0; i < surf.m_nTriangles; i++)
  {
    int v = surf.m_piTriangleIndices[i * 3];
    glNormal3fv(surf.m_pvec3dNormals[v]);
    glVertex3fv(surf.m_ppt3dVertices[v]);

    v = surf.m_piTriangleIndices[i * 3 + 1];
    glNormal3fv(surf.m_pvec3dNormals[v]);
    glVertex3fv(surf.m_ppt3dVertices[v]);

    v = surf.m_piTriangleIndices[i * 3 + 2];
    glNormal3fv(surf.m_pvec3dNormals[v]);
    glVertex3fv(surf.m_ppt3dVertices[v]);
  }
  glEnd();

  glDepthMask(GL_TRUE);
  glDisable(GL_BLEND);
}


void Renderer::drawDensity()
{
  float tdens = 1.0f;
  const Volume<float> &dens = skel.getDensity(tdens);

  const int width = dens.getWidth(), height = dens.getHeight(),
            depth = dens.getDepth();

  float col[4];
  const float dens_scale = 20.0f;//0.5f*tdens; //2.0f

  for (int i = 1; i < depth - 1; i++)
    for (int j = 1; j < height - 1; j++)
      for (int k = 1; k < width - 1; k++)
      {
        if (dens[i][j][k] < 1.99f) continue;

        float ndens = dens[i][j][k] / dens_scale;

        if (ndens < imp_thr) continue;

        float2rgb(ndens, col);
        glColor3fv(col);

        glTranslatef(k, j, i);
        glutSolidCube(1.0f);
        glTranslatef(-k, -j, -i);
      }
}

void Renderer::drawArrow(Point3d point, Vector direction, float len)
{
  float mat[16];
  mat[3] = mat[7] = mat[11] = mat[12] = mat[13] = mat[14] = 0;
  mat[15] = 1;

  const float w_upx = 0.0f, w_upy = 1.0f, w_upz = 0.0f; //world up vector (0,1,0)
  int i = point.z, j = point.y, k = point.x;

  //view vector
  const Vector &vec = direction;
  float vx = vec.u, vy = vec.v, vz = vec.w;
  float mag = sqrtf(vx * vx + vy * vy + vz * vz);
  if (mag < 1e-7f) return;
  vx /= mag;
  vy /= mag;
  vz /= mag;

  float a, b, c;
  Point3d vv(vx, vy, vz);
  a = fabs(vv.dot(Point3d(1, 0, 0)));
  b = fabs(vv.dot(Point3d(0, 1, 0)));
  c = fabs(vv.dot(Point3d(0, 0, 1)));

  glColor3f(a, b, c);

  //up vector
  float up_proj = vy;
  float v_upx = w_upx - up_proj * vx;
  float v_upy = w_upy - up_proj * vy;
  float v_upz = w_upz - up_proj * vz;
  mag = sqrtf(v_upx * v_upx + v_upy * v_upy + v_upz * v_upz);
  if (mag < 1e-7f) return;
  v_upx /= mag;
  v_upy /= mag;
  v_upz /= mag;

  //side vector
  float v_rx = -vy * v_upz + vz * v_upy;
  float v_ry = -vz * v_upx + vx * v_upz;
  float v_rz = -vx * v_upy + vy * v_upx;

  //fill-in rotation part
  mat[0] = v_rx;
  mat[1] = v_ry;
  mat[2] = v_rz;
  mat[4] = v_upx;
  mat[5] = v_upy;
  mat[6] = v_upz;
  mat[8] = vx;
  mat[9] = vy;
  mat[10] = vz;

  //arrow head
  glPushMatrix();
  float arrow_x = k + 0.6f * vx * len, arrow_y = j + 0.6f * vy * len,
    arrow_z = i + 0.6f * vz * len;
  glTranslatef(arrow_x, arrow_y, arrow_z);

  glMultMatrixf(mat);
  gluCylinder(quad, 0.30f * len, 0.0f, 0.45f * len, 10, 10);
  glPopMatrix();

  //arrow body
  glPushMatrix();
  glTranslatef(k, j, i);
  glMultMatrixf(mat);
  gluCylinder(quad, 0.12f * len, 0.12f * len, 0.6f * len, 10, 10);
  glPopMatrix();
}

void Renderer::drawArrow(Point3d point, float len)
{
  const Volume<Vector> &v = skel.getV();
  int i = point.z, j = point.y, k = point.x;

  // Velocity vector
  const Vector &vec = v(k, j, i);

  drawArrow(point, vec, len);
}

void Renderer::drawVelocity(int step, float len)
{
  const Volume<Vector> &v = skel.getV();
  const std::vector<Point3d>& skel_points = skeletonModel->GetSkelPoints();
  glDisable(GL_LIGHTING);

  float mat[16];
  mat[3] = mat[7] = mat[11] = mat[12] = mat[13] = mat[14] = 0;
  mat[15] = 1;

  //const float len = 1.0f; //arrow length
  const float w_upx = 0.0f, w_upy = 1.0f, w_upz = 0.0f; //world up vector (0,1,0)

  for (size_t id = 0, NP = skel_points.size(); id < NP; ++id)
  {
    const Point3d &skp = skel_points[id];
    int i = skp.z, j = skp.y, k = skp.x;

    //view vector
    const Vector &vec = v(k, j, i);

    drawArrow(skp, vec, len);
  }

  glEnable(GL_LIGHTING);
}


void Renderer::drawThinGrad(int step, float len)
{
  //spline kernel
  static const float kernel[10] = {1.0f, 4.0f, 1.0f, 4.0f, 16.0f, 4.0f, 1.0f, 4.0f, 1.0f, 72.0f};

  float tdens = 1.0f;
  const Volume<float> &dens = skel.getDensity(tdens);

  const int width = dens.getWidth(), height = dens.getHeight(),
            depth = dens.getDepth();

  float mat[16], col[4];
  mat[3] = mat[7] = mat[11] = mat[12] = mat[13] = mat[14] = 0;
  mat[15] = 1;

  //const float len = 1.0f; //arrow length
  const float w_upx = 0.0f, w_upy = 1.0f, w_upz = 0.0f; //world up vector (0,1,0)

  const Volume<byte> &gimg = skel.getThinImage();


  for (int i = 1; i < depth - 1; i += step)
    for (int j = 1; j < height - 1; j += step)
      for (int k = 1; k < width - 1; k += step)
      {

        //if(imp[i][j][k]<imp_thr) continue;
        //if(dens[i][j][k]/tdens<imp_thr) continue;

        float gx = 0.0f, gy = 0.0f, gz = 0.0f, kval;


        if (gimg[i][j][k]) continue;

        int idx = 0;
        for (int n = -1; n <= 1; n++)
          for (int m = -1; m <= 1; m++)
          {
            kval = kernel[idx++];
            gx += kval * (gimg[i + n][j + m][k + 1] - gimg[i + n][j + m][k - 1]);
            gy += kval * (gimg[i + n][j + 1][k + m] - gimg[i + n][j - 1][k + m]);
            gz += kval * (gimg[i + 1][j + n][k + m] - gimg[i - 1][j + n][k + m]);
          }

        //view vector
        float vx = gx, vy = gy, vz = gz;
        float mag = sqrtf(vx * vx + vy * vy + vz * vz);
        if (mag < 1e-7f) continue;
        vx /= mag;
        vy /= mag;
        vz /= mag;


        float2rgb(dens[i][j][k]*mag / 5.0f, col);
        glColor3fv(col);

        //up vector
        float up_proj = vy;
        float v_upx = w_upx - up_proj * vx;
        float v_upy = w_upy - up_proj * vy;
        float v_upz = w_upz - up_proj * vz;
        mag = sqrtf(v_upx * v_upx + v_upy * v_upy + v_upz * v_upz);
        if (mag < 1e-7f) continue;
        v_upx /= mag;
        v_upy /= mag;
        v_upz /= mag;

        //side vector
        float v_rx = -vy * v_upz + vz * v_upy;
        float v_ry = -vz * v_upx + vx * v_upz;
        float v_rz = -vx * v_upy + vy * v_upx;

        //fill-in rotation part
        mat[0] = v_rx;
        mat[1] = v_ry;
        mat[2] = v_rz;
        mat[4] = v_upx;
        mat[5] = v_upy;
        mat[6] = v_upz;
        mat[8] = vx;
        mat[9] = vy;
        mat[10] = vz;

        //arrow head
        glPushMatrix();
        float arrow_x = k + 0.6f * vx * len, arrow_y = j + 0.6f * vy * len,
              arrow_z =  i + 0.6f * vz * len;
        glTranslatef(arrow_x, arrow_y, arrow_z);

        glMultMatrixf(mat);

        gluCylinder(quad, 0.25f * len, 0.0f, 0.41f * len, 6, 3);
        glPopMatrix();

        //arrow body
        glPushMatrix();
        glTranslatef(k, j, i);

        glMultMatrixf(mat);

        gluCylinder(quad, 0.12f * len, 0.12f * len, 0.6f * len, 6, 3);
        glPopMatrix();
      }
}


