﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using ICSharpCode.NRefactory.Semantics;
using ICSharpCode.NRefactory.TypeSystem.Implementation;
using ICSharpCode.NRefactory.TypeSystem;

namespace CSharpDeps.Scopes
{
    public class Method
    {
        public Class Class { get; private set; }

        public string Name { get; private set; }

        public Namespace Namespace
        {
            get { return Class.Namespace; }
        }

        public string FullName
        {
            get { return string.Format("{0}.{1}", Class.FullName, Name); }
        }

        protected Dictionary<string, Callee> invocations = new Dictionary<string, Callee>();

        public Dictionary<string, Callee> Invocations
        {
            get { return invocations; }
        }

        protected Dictionary<string, AccessedMember> accesses = new Dictionary<string, AccessedMember>();

        public Dictionary<string, AccessedMember> MemberAccesses
        {
            get { return accesses; }
        }

        public Method(Class klass, string name)
        {
            Class = klass;
            Name = name;
        }

        internal void AddInvocation(InvocationResolveResult invocation)
        {
            var defaultMethod = invocation.Member as DefaultResolvedMethod;
            var specializedMethod = invocation.Member as SpecializedMethod;

            if (defaultMethod == null && specializedMethod == null) return;

            var parms = defaultMethod != null
                ? ParameterInfo(defaultMethod)
                : ParameterInfo(specializedMethod);

            var callee = string.Format("{0}{1}", invocation.Member.FullName, parms);
            
            if (invocations.ContainsKey(callee))
            {
                invocations[callee].Invocations++;
            }
            else
            {
                invocations[callee] = new Callee
                {
                    Class = GetClass(invocation.Member),
                    Name = string.Format("{0}{1}", invocation.Member.Name, parms),
                    Invocations = 1,
                };
            }
        }

        public static string ParameterInfo(DefaultResolvedMethod method)
        {
            return ParameterInfo(method.Parameters, method.TypeParameters);
        }

        public static string ParameterInfo(SpecializedMethod method)
        {
            return ParameterInfo(method.Parameters, method.TypeParameters);
        }

        public static string ParameterInfo(IList<IParameter> parameters, IList<ITypeParameter> types)
        {
            var parms = string.Join(", ", parameters.Select(p => p.Type.FullName));

            if (types.Count > 0)
            {
                var typs = string.Join(", ", types.Select(p => p.FullName));

                return string.Format("<{0}>({1})", typs, parms);
            }
            else
            {
                return string.Format("({0})", parms);
            }
        }

        internal void AddMemberAccess(MemberResolveResult member)
        {
            var name = member.Member.FullName;

            if (accesses.ContainsKey(name))
            {
                accesses[name].Accesses++;
            }
            else
            {
                accesses[name] = new AccessedMember
                {
                    Class = GetClass(member.Member),
                    Name = member.Member.Name,
                    Accesses = 1,
                };
            }
        }

        internal ClassReference GetClass(IMember member)
        {
            if (member.DeclaringTypeDefinition != null)
            {
                return new ClassReference
                {
                    Namespace = member.Namespace,
                    Name = member.DeclaringTypeDefinition.Name,
                    Type = Scopes.Class.TranslateKind(member.DeclaringTypeDefinition.Kind),
                };
            }
            else
            {
                return new ClassReference
                {
                    Namespace = "",
                    Name = "?",
                    Type = Scopes.Class.Types.Class,
                };
            }
        }
    }

    public class Callee
    {
        public ClassReference Class { get; set; }

        public string Name { get; set; }

        public int Invocations { get; set; }

        public string FullName
        {
            get { return string.Format("{0}.{1}", Class.FullName, Name).Trim('.'); }
        }
    }

    public class AccessedMember
    {
        public ClassReference Class { get; set; }

        public string Name { get; set; }

        public int Accesses { get; set; }

        public string FullName
        {
            get { return string.Format("{0}.{1}", Class.FullName, Name).Trim('.'); }
        }
    }
}
