
import wx
import sta_globals
import sta_plugin
import cushions
import sta_utils
import time
import calendar

from sta_gl			import CanvasBase
from sta_gl			import glPanel
from OpenGL.GL			import *
from OpenGL.GLUT		import *
from OpenGL.GLU		import *

#/////////////////////////////////////////////////////////////////////////
#														Mixer Canvas
#/////////////////////////////////////////////////////////////////////////

class AttrEvolutionView:

	def __init__(self, p_cMainApp, p_sClass, *args, **kwds):

		#--------------------------------------------------------------- GUI ---
		self.cMainFrame				= p_cMainApp.mainFrame

		self.g_panelGLSupport		  = glPanel(self.cMainFrame.window_graphs)
													# Attribute evolution view
		self.g_sAttrEvolutionFactor = wx.Slider(self.cMainFrame.window_graphs, -1, 100, 1, 100, style = wx.SL_VERTICAL | wx.SL_INVERSE )
													# Attribute value normalizer
		self.g_sAttrEvolutionFactor.SetMinSize((15,-1))

		self.sizerAttributeView		= wx.BoxSizer(wx.HORIZONTAL)
		self.sizerAttributeView.Add(self.g_panelGLSupport,1, wx.EXPAND, 0)
		self.g_GLPanel = AttrCanvas(p_sClass,p_cMainApp,self.g_panelGLSupport)
		self.sizerAttributeView.Add(self.g_sAttrEvolutionFactor,0, wx.EXPAND|wx.ADJUST_MINSIZE, 0)

		self.cMainFrame.sizerAttributeView.Add(self.sizerAttributeView,1,wx.EXPAND | wx.TOP,3)
		self.cMainFrame.sizerAttributeView.Layout()

		#------------------------------------------------------- Interaction ---
		self.g_sAttrEvolutionFactor.Bind(wx.EVT_SLIDER,self.cbAttrEvolutionFactor)

#------------------------------------------------------------------- CleanUp ---
	def CleanUp(self):
		self.g_GLPanel = 0

		self.sizerAttributeView.Detach(self.g_sAttrEvolutionFactor)
		self.g_sAttrEvolutionFactor.Destroy()

		self.sizerAttributeView.Detach(self.g_panelGLSupport)
		self.g_panelGLSupport.Destroy()

		self.cMainFrame.sizerAttributeView.Remove(self.sizerAttributeView)
		self.cMainFrame.sizerAttributeView.Layout()

#------------------------------------------ Attribute evolution scale factor ---
	def cbAttrEvolutionFactor(self,evt):
		self.g_GLPanel.fAttrValueFactor = 100.0 / self.g_sAttrEvolutionFactor.GetValue()
		self.g_GLPanel.RefreshEx()

#------------------------------------------------------------------- Refresh ---
	def Refresh(self):
		self.g_GLPanel.RefreshEx()

#---------------------------------------------------------------------- Show ---
	def Show(self,p_bState):
		if p_bState:
			self.cMainFrame.sizerAttributeView.Show(self.sizerAttributeView,True)
			self.cMainFrame.sizerAttributeView.Layout()
			self.g_GLPanel.RefreshEx()
		else:
			self.cMainFrame.sizerAttributeView.Show(self.sizerAttributeView,False)
			self.cMainFrame.sizerAttributeView.Layout()

class AttrCanvas(CanvasBase):
	def __init__(self, p_sClass, p_cMainApp, *args, **kwds):

		self.cMainApp			= p_cMainApp

		CanvasBase.__init__(self, *args, **kwds)
		self.bUpdate			= True				# Reconfigure draw

		self.dData				= {}				# Contains the table info to visualize (attributes vs. intervals)
		self.cPlugin			= sta_plugin.getPluginByClass(sta_globals.lPlugins_Project,p_sClass)
		self.iNrAttrs			= 0					# Number of attributes to display
		self.iNrIntervals		= 0					# Number of time intervals to display
		self.lIntervals			= []				# Currently selected time display interval
		self.intervals			= []   				# List of time intervals at various levels
		self.fDataScale			= 1					# The vertical scale for data values

		self.fItemHeight		= 0					# Height of one attribute
		self.iBorderX			= 20				# Label border size
		self.lIntervalSize		= [86400,604800,16934400,50803200,101606400,203212800]
		self.iVScaleType		= 1					# Y axis ( 0 = nr versions, 1 = nr files, 2,3,4,5,6 = values)

		self.iVScaleName		= ['Commits','Files','Total','Average']
		self.iAttrTimeInterval	= 2					# The interval choice
		self.fAttrValueFactor	= 1					# The attribute value magnification factor 1
		self.cAttrCushions		= cushions.CCushion()	# Attribute evolution panel cushions
		self.iDisplayType		= 0					# Display type

		#----------------------------- Pop-up menu
		self.cMenu = wx.Menu("Display type")
		self.cMenu.AppendCheckItem(201,"Bar chart")				# Display type = 0
		self.cMenu.AppendCheckItem(202,"Graph")					# Display type = 1
		self.cMenu.AppendCheckItem(203,"Intensity map")			# Display type = 2
		self.cMenu.AppendCheckItem(204,"Rainbow map")			# Display type = 3
		self.cMenu.AppendCheckItem(205,"Flow")					# Display type = 4
		self.cMenu.AppendSeparator()
		self.cMenu.AppendCheckItem(210,"Commit count")			# Y axis = number of commits
		self.cMenu.AppendCheckItem(211,"File count")			# Y axis = number of files
		self.cMenu.AppendCheckItem(212,"Total value")			# Y axis = total value
		self.cMenu.AppendCheckItem(213,"Average value")			# Y axis = average value
		self.cMenu.AppendSeparator()
		self.cMenu.Append(230,"Take snaphot")					# Take snapshot
		self.cMenu.Append(231,"Save values")					# Save values

		self.Bind(wx.EVT_MENU, self.MenuDisplayType, id=201)
		self.Bind(wx.EVT_MENU, self.MenuDisplayType, id=202)
		self.Bind(wx.EVT_MENU, self.MenuDisplayType, id=203)
		self.Bind(wx.EVT_MENU, self.MenuDisplayType, id=204)
		self.Bind(wx.EVT_MENU, self.MenuDisplayType, id=205)
		self.Bind(wx.EVT_MENU, self.VScaleType, id=210)
		self.Bind(wx.EVT_MENU, self.VScaleType, id=211)
		self.Bind(wx.EVT_MENU, self.VScaleType, id=212)
		self.Bind(wx.EVT_MENU, self.VScaleType, id=213)
		self.Bind(wx.EVT_MENU, self.cbSnapShot, id=230)
		self.Bind(wx.EVT_MENU, self.cbSaveValues, id=231)

		if self.cPlugin.bHasMetricValue:
			self.iVScaleType	= 2
			mi = self.cMenu.FindItemById(210)
			mi.Enable(False)
		mi = self.cMenu.FindItemById(201)
		mi.Check(True)
		mi = self.cMenu.FindItemById(210+self.iVScaleType)
		mi.Check(True)
		#-----------------------------------------

	def InitGL(self):
		glMatrixMode(GL_MODELVIEW)
		glClearColor(1, 1, 1, 0)
		cushions.generateMainTextures(self.cAttrCushions)
#===============================================================================
#																 Menu events
#===============================================================================
	def MenuDisplayType(self,evt):
		self.iDisplayType = evt.GetId() - 201
		#--------------------- Uncheck items ---
		for i in range(5):
			mi = self.cMenu.FindItemById(201+i)
			if mi.IsChecked():
				mi.Check(False)
		#-------------------- Check selected ---
		mi = self.cMenu.FindItemById(201+self.iDisplayType)
		mi.Check(True)

		self.Refresh()

	def VScaleType(self,evt):
		self.iVScaleType = evt.GetId() - 210
		for i in range(4):
			mi = self.cMenu.FindItemById(210+i)
			if mi.IsChecked():
				mi.Check(False)
		mi = self.cMenu.FindItemById(210+self.iVScaleType)
		mi.Check(True)

		self.Refresh()

	def cbSaveValues(self,evt):
		l_cDlg = wx.FileDialog(self.cGUI,style = wx.FD_SAVE, wildcard = "Comma separated values files (*.csv)|*.csv")
		if (l_cDlg.ShowModal() == wx.ID_OK):
			sFilePath = l_cDlg.GetPath()

			fileHandle = open(sFilePath,'w')
			fileHandle.write("%s (%s)\n"%(self.cPlugin.sName,self.iVScaleName[self.iVScaleType]))
			fileHandle.write("------------------\n\n")

			# Header
			l_sData = self.cPlugin.cGUIList.GetSelections()
			l_sHeader = 'Date'
			for i in range(self.iNrAttrs):
				if (self.iVScaleType > 1):
					l_sHeader += ',Value'
				else:
					l_sHeader += (',%s'%l_sData[i])
			fileHandle.write("%s\n"%l_sHeader)

			for l_iInterval in range(self.iNrIntervals):

				l_sRecord = sta_utils.getDate(self.lIntervals[l_iInterval])

				for i in range(self.iNrAttrs):

					if (self.iVScaleType > 1):
						l_iValue = 'Value'
					else:
						l_iValue = self.cPlugin.lIdx[i]

					l_fValue = self.dData[l_iValue][l_iInterval] * self.fDataScale * self.fAttrValueFactor
					if (self.iVScaleType in [0,1]):
						l_fValue = int(l_fValue)
					l_sValue = str(l_fValue)

					l_sRecord += ',%s'%l_sValue

				fileHandle.write("%s\n"%l_sRecord)

			fileHandle.close()
		l_cDlg.Destroy()
#===============================================================================
#																 Mouse Events
#===============================================================================
	def OnMouseWheel(self,evt):

		#---------------------------------------------------- Time interval ---
		if evt.m_controlDown:
			if (evt.m_wheelRotation < 0):
				if (self.iAttrTimeInterval > 0):
					if (self.lIntervalSize[self.iAttrTimeInterval-1] > sta_globals.fUnitX * 5):
						self.iAttrTimeInterval = self.iAttrTimeInterval - 1
			else:
				if (self.iAttrTimeInterval < len(self.intervals)-1):
					self.iAttrTimeInterval = self.iAttrTimeInterval + 1
			self.Refresh()

	def OnRightMouseDown(self, evt):
		self.PopupMenu(self.cMenu)

	def OnMouseMotion(self, evt):

		self.x, self.y = evt.GetPosition()
		self.x = max(self.x,0)
		self.x = min(self.x,self.size.width)
		self.y = max(self.y,0)
		self.y = min(self.y,self.size.height)

		l_iValue = self.fDataScale * (self.size.height - self.y)/self.size.height
		l_iTime = self.x * sta_globals.fUnitX + sta_globals.fOffsetX
		l_sDate = sta_utils.getDate(l_iTime)

		self.cMainApp.mainFrame.statusbar.SetStatusText("",0)
		self.cMainApp.mainFrame.statusbar.SetStatusText("",3)
		self.cMainApp.mainFrame.statusbar.SetStatusText("",4)
		if (self.iVScaleType > 1):
			self.cMainApp.mainFrame.statusbar.SetStatusText(str(round(l_iValue,3)),5)
		else:
			self.cMainApp.mainFrame.statusbar.SetStatusText("",5)
		self.cMainApp.mainFrame.statusbar.SetStatusText(l_sDate,6)
#===============================================================================
#																		OnDraw
#===============================================================================

	def OnDraw(self):

		if self.bRendering:
			return
		self.glListSetup()

		if len(sta_globals.lSelectedFiles)>0:
		#--------------------------------------------- Update configuration ---
			if (self.bUpdate):
				self.UpdateTime()
				self.bUpdate = False
		#----------------------------------------- Start drawing (anchored) ---
		glTranslatef(0,self.size.height,0)
		glPushMatrix()
		glScalef(1,-1,1)
		glColor3f(0.9,1.0,0.9)
		glRectf(0,0,self.size.width,self.size.height)

		# TODO This should not be computed all the time but only when attribute selection changes (use observers?)
		self.UpdateData()

		#--------------------------------------------- Draw data (floating) ---
		if (self.iNrAttrs > 0):

			self.fItemHeight = float(self.size.height) / self.iNrAttrs

			if self.iDisplayType == 0:
				self.drawBarCharts()

			if self.iDisplayType == 1:
				self.drawGraph()

			if self.iDisplayType == 2:
				self.drawIntensityMap()

			if self.iDisplayType == 3:
				self.drawRainbowMap()

			if self.iDisplayType == 4:
				self.drawFlow()

			glDisable(GL_BLEND)

		#--------------------------------------- Draw time scale (floating) ---
		if sta_globals.bAttrGrid:
			glColor3f(0.4,0.6,0.4)
			if self.iNrIntervals > 1:
				if (self.lIntervals[1] - self.lIntervals[0]) > 3 *  sta_globals.fUnitX:
					for i in range(self.iNrAttrs):
						for j in range(self.iNrIntervals-1):
							glRectf(\
									self.lIntervals[j],\
									i*self.fItemHeight,\
									self.lIntervals[j] + sta_globals.fUnitX,\
									(i+1)*self.fItemHeight)
		#-------------------------------------- Draw time labels (floating) ---
		if sta_globals.bAttrTimeLabels:
			for i in self.intervals[5]:
				l_lTime = time.gmtime(i)
				glColor3f(0.5,0.2,0.2)
				glRectf(i-sta_globals.fUnitX,0,i+2*sta_globals.fUnitX,20)
				self.drawGlowString(i+5*sta_globals.fUnitX,20,str(l_lTime[0]))
			l_bShowMonths = sta_globals.fRatioX > (1.3 / 100000)
			for i in self.intervals[2]:
				l_lTime = time.gmtime(i)
				glColor3f(0.5,0.2,0.2)
				glRectf(i-sta_globals.fUnitX,0,i+sta_globals.fUnitX,10)
				if l_bShowMonths:
					self.drawGlowString(i+5*sta_globals.fUnitX,10,sta_globals.sMonths[l_lTime[1]-1])
		#-------------------------------------------- Print anchored labels ---
		glPopMatrix()
		glScalef(1,-1,1)
		#-------------------------------- Print attribute labels (anchored) ---
		if (self.iNrAttrs > 0):
			if self.iVScaleType in [0,1]:
				# colors
				glBlendFunc(GL_DST_ALPHA,GL_ZERO)
				glEnable(GL_BLEND)
				for i in range(self.iNrAttrs):
					l_lColor = self.cPlugin.cGUIList.lColors[self.cPlugin.cGUIList.dColors[self.cPlugin.lData[self.cPlugin.lIdx[i]]]]
					glColor3fv(l_lColor)
					glRectf(0,i*self.fItemHeight,self.iBorderX,(i+1)*self.fItemHeight)
				glDisable(GL_BLEND)
				# text
				l_sData = self.cPlugin.cGUIList.GetSelections()
				if sta_globals.bAttrLabels:
					for i in range(self.iNrAttrs):
						if self.fItemHeight > 10:
							self.drawGlowString(25,(i+1)*self.fItemHeight - 4,l_sData[i])

		#---------------------------- Print vertical scale label (anchored) ---
		if sta_globals.bAttrVScale:
				l_fX = 0
				l_fY = 0
				glColor3f(0.5,0,0)
				glRectf(l_fX,l_fY,l_fX+81,l_fY+12)
				glColor3f(0.9,0.9,0.7)
				glRectf(l_fX+1,l_fY+1,l_fX+80,l_fY+11)
				l_sDataScale = str(self.fDataScale)+' '+self.iVScaleName[self.iVScaleType]
				self.drawGlowString(l_fX+6,l_fY+10,l_sDataScale)

		#------------------------------------ Print filter label (anchored) ---
		if sta_globals.bAttrNameLabel:
				l_fX = self.size.width - 80
				l_fY = self.size.height - 12
				glColor3f(0,0,0.5)
				glRectf(l_fX-1,l_fY,l_fX+80,l_fY+12)
				glColor3f(0.9,0.9,0.7)
				glRectf(l_fX,l_fY+1,l_fX+79,l_fY+11)
				self.drawGlowString(l_fX+5,l_fY+10,self.cPlugin.sName)

		#------------------------------------------------------ End drawing ---
		self.glListRun(True)

#-------------------------------------------------------------------------------
#																	Bar charts
#-------------------------------------------------------------------------------
	def drawSegments(self):
		#-------------------------------- Draw bars and cushions ---
		glBlendFunc(GL_ONE,GL_ZERO)
		glEnable(GL_BLEND)
		for i in range(self.iNrAttrs):
			cushions.drawCushion1DH(0,i*self.fItemHeight,self.size.width,(i+1)*self.fItemHeight,self.cAttrCushions,2,True)
		glDisable(GL_BLEND)

		glBlendFunc(GL_DST_ALPHA,GL_ZERO)
		glEnable(GL_BLEND)

		glColor4f(0.9,1.0,0.9,1.0)
		for i in range(self.iNrAttrs):
			glRectf(self.iBorderX,i*self.fItemHeight,self.size.width,(i+1)*self.fItemHeight)

		#---------------------------- Set-up floating cooridnates ---
		glPopMatrix()
		glPushMatrix()

		glScalef(sta_globals.fRatioX,-1,1)
		glTranslatef(-sta_globals.fOffsetX + self.iBorderX * sta_globals.fUnitX,0,0)

	def drawBarCharts(self):
		self.drawSegments()
		for i in range(self.iNrAttrs):
			l_iValue = self.GetDataPointer(i)
			for j in range(self.iNrIntervals-1):

				glColor4f(0.4,0.7,0.4,1.0)
				l_fDisplayValue = self.dData[l_iValue][j] * self.fAttrValueFactor
				if l_fDisplayValue > 1:
					l_fDisplayValue = (l_fDisplayValue - 1) / self.fAttrValueFactor
					glRectf(\
						self.lIntervals[j],\
						i*self.fItemHeight,\
						self.lIntervals[j+1],\
						(i+1)*self.fItemHeight)
					glColor3f(0.6,0.9,0.6)

				glRectf(\
					self.lIntervals[j],\
					(i+1-l_fDisplayValue)*self.fItemHeight,\
					self.lIntervals[j+1],\
					(i+1)*self.fItemHeight)
#-------------------------------------------------------------------------------
#																		Graph
#-------------------------------------------------------------------------------
	def drawGraph(self):

		self.drawSegments()
		#----------------- First level ---
		for i in range(self.iNrAttrs):
			l_iValue = self.GetDataPointer(i)

			glBegin(GL_QUAD_STRIP)

			if self.iNrIntervals > 0:
				glVertex2f(self.lIntervals[0],(i+1)*self.fItemHeight)
				glVertex2f(self.lIntervals[0],(i+1)*self.fItemHeight)

			for j in range(self.iNrIntervals-1):

				glColor4f(0.4,0.7,0.4,1.0)
				l_fDisplayValue = self.dData[l_iValue][j] * self.fAttrValueFactor

				if l_fDisplayValue > 1:
					l_fDisplayValue = 1

				l_fX = self.lIntervals[j]+float(self.lIntervals[j+1] - self.lIntervals[j])/2
				glVertex2f(l_fX,(i+1)*self.fItemHeight)
				glVertex2f(l_fX,(i+1-l_fDisplayValue)*self.fItemHeight)

			glVertex2f(sta_globals.iSelectedFilesEndTime,(i+1)*self.fItemHeight)
			glVertex2f(sta_globals.iSelectedFilesEndTime,(i+1-l_fDisplayValue)*self.fItemHeight)
			glEnd()

		#----------------- Second level ---
		for i in range(self.iNrAttrs):
			l_iValue = self.GetDataPointer(i)

			glBegin(GL_QUAD_STRIP)

			if self.iNrIntervals > 0:
				glVertex2f(self.lIntervals[0],(i+1)*self.fItemHeight)
				glVertex2f(self.lIntervals[0],(i+1)*self.fItemHeight)

			for j in range(self.iNrIntervals-1):

				glColor3f(0.6,0.9,0.6)
				l_fDisplayValue = self.dData[l_iValue][j] * self.fAttrValueFactor

				if l_fDisplayValue > 1:
					l_fDisplayValue = (l_fDisplayValue - 1) / self.fAttrValueFactor
				else:
					l_fDisplayValue = 0

				l_fX = self.lIntervals[j]+float(self.lIntervals[j+1] - self.lIntervals[j])/2
				glVertex2f(l_fX,(i+1)*self.fItemHeight)
				glVertex2f(l_fX,(i+1-l_fDisplayValue)*self.fItemHeight)

			glVertex2f(sta_globals.iSelectedFilesEndTime,(i+1)*self.fItemHeight)
			glVertex2f(sta_globals.iSelectedFilesEndTime,(i+1-l_fDisplayValue)*self.fItemHeight)
			glEnd()

#-------------------------------------------------------------------------------
#																	Intensity map
#-------------------------------------------------------------------------------
	def drawIntensityMap(self):

		self.drawSegments()

		glDisable(GL_BLEND)
		glBlendFunc(GL_ZERO,GL_SRC_ALPHA)
		glEnable(GL_BLEND)

		for i in range(self.iNrAttrs):
			l_iValue = self.GetDataPointer(i)
			for j in range(self.iNrIntervals-1):

				l_fDisplayValue = self.dData[l_iValue][j] * self.fAttrValueFactor
				if l_fDisplayValue > 1:
					l_fDisplayValue = 1

				l_fFactor = (1-l_fDisplayValue)
				glColor4f(1.0,1.0,1.0,l_fFactor)

				glRectf(\
						self.lIntervals[j],\
						i*self.fItemHeight,\
						self.lIntervals[j+1],\
						(i+1)*self.fItemHeight)
		glDisable(GL_BLEND)
		glBlendFunc(GL_DST_ALPHA,GL_ZERO)
		glEnable(GL_BLEND)

#-------------------------------------------------------------------------------
#																	Rainbow map
#-------------------------------------------------------------------------------
	def drawRainbowMap(self):

		self.drawSegments()

		glDisable(GL_BLEND)
		glBlendFunc(GL_DST_ALPHA,GL_ZERO)
		glEnable(GL_BLEND)

		for i in range(self.iNrAttrs):
			l_iValue = self.GetDataPointer(i)
			for j in range(self.iNrIntervals-1):

				l_fDisplayValue = self.dData[l_iValue][j] * self.fAttrValueFactor
				if l_fDisplayValue > 1:
					l_fDisplayValue = 1

				if l_fDisplayValue == 0:
					glColor3f(0.9,1.0,0.9)
				else:
					glColor3fv(sta_utils.getRainbowColor(l_fDisplayValue))

				glRectf(\
						self.lIntervals[j],\
						i*self.fItemHeight,\
						self.lIntervals[j+1],\
						(i+1)*self.fItemHeight)

#-------------------------------------------------------------------------------
#																		Flow
#-------------------------------------------------------------------------------
	def drawFlow(self):

		glColor4f(0.9,1.0,0.9,1.0)
		for i in range(self.iNrAttrs):
			glRectf(self.iBorderX,i*self.fItemHeight,self.size.width,(i+1)*self.fItemHeight)

		#------------------------------------------ Find offsets ---
		l_lOffsets = [0 for i in range(self.iNrIntervals)]
		for i in range(self.iNrAttrs):
			l_iValue = self.GetDataPointer(i)
			for j in range(self.iNrIntervals):
				l_fDisplayValue = self.dData[l_iValue][j] * self.fAttrValueFactor
				if l_fDisplayValue > 1:
						l_fDisplayValue = 1
				l_lOffsets[j] = l_lOffsets[j] + l_fDisplayValue

		l_fMaxOffset	= max(l_lOffsets)
		if l_fMaxOffset == 0:
			return

		l_fYRatio	   = float(self.iNrAttrs)/l_fMaxOffset
		for j in range(self.iNrIntervals):
			l_lOffsets[j] = (l_fMaxOffset - l_lOffsets[j]) / 2

		#----------------------------------- Compute data points ---
		l_lDataPoints = [[] for k in range(self.iNrIntervals+1)]
		for j in range(self.iNrIntervals-1):

			l_lDataPoints[j+1] = [0 for k in range(self.iNrAttrs+1)]
			l_fX = self.lIntervals[j]+float(self.lIntervals[j+1] - self.lIntervals[j])/2
			l_fOffset1 = l_lOffsets[j]

			for i in range(self.iNrAttrs):
				l_iValue = self.GetDataPointer(i)
				l_lDataPoints[j+1][i] = (l_fX,l_fOffset1)
				l_fDisplayValue = self.dData[l_iValue][j] * self.fAttrValueFactor
				if l_fDisplayValue > 1:
						l_fDisplayValue = 1
				l_fOffset1 = l_fOffset1 + l_fDisplayValue
			l_lDataPoints[j+1][i+1] = (l_fX,l_fOffset1)

		# TODO: Check if this is possible (indexies exist)
		l_lDataPoints[ 0]   = [0 for k in range(self.iNrAttrs+1)]
		l_lDataPoints[-1]   = [0 for k in range(self.iNrAttrs+1)]
		for i in range(self.iNrAttrs+1):
			l_lDataPoints[ 0][i] = (self.lIntervals[ 0],l_lDataPoints[ 1][i][1])
			l_lDataPoints[-1][i] = (self.lIntervals[-1],l_lDataPoints[-2][i][1])

		#---------------------------------------- Draw evolution ---
		glPopMatrix()
		glPushMatrix()

		glScalef(sta_globals.fRatioX,-1,1)
		glTranslatef(-sta_globals.fOffsetX + self.iBorderX * sta_globals.fUnitX,0,0)

		for i in range(self.iNrAttrs):

			l_iValue = self.GetDataPointer(i)
			if self.iVScaleType <=1:
				l_lColor = self.cPlugin.cGUIList.lColors[self.cPlugin.cGUIList.dColors[self.cPlugin.lData[self.cPlugin.lIdx[i]]]]
			else:
				l_lColor = (0.4,0.7,0.4)
			glColor3fv(l_lColor)

			for j in range(self.iNrIntervals):

				l_fX1_u = l_lDataPoints[j  ][i  ][0]
				l_fX1_d = l_lDataPoints[j  ][i+1][0]
				l_fY1_u = l_lDataPoints[j  ][i  ][1]
				l_fY1_d = l_lDataPoints[j  ][i+1][1]
				l_fX2_u = l_lDataPoints[j+1][i  ][0]
				l_fX2_d = l_lDataPoints[j+1][i+1][0]
				l_fY2_u = l_lDataPoints[j+1][i  ][1]
				l_fY2_d = l_lDataPoints[j+1][i+1][1]

				# Cushions
				glBlendFunc(GL_ONE,GL_ZERO)
				glEnable(GL_BLEND)
				glBindTexture(GL_TEXTURE_1D,self.cAttrCushions.tex[2])
				glEnable(GL_TEXTURE_1D)
				glTexEnvf(GL_TEXTURE_ENV,GL_TEXTURE_ENV_MODE,GL_REPLACE)

				glBegin(GL_QUADS)
				glTexCoord1i(0)
				glVertex2f(l_fX1_u,l_fY1_u*self.fItemHeight*l_fYRatio)
				glTexCoord1i(0)
				glVertex2f(l_fX2_u,l_fY2_u*self.fItemHeight*l_fYRatio)
				glTexCoord1i(1)
				glVertex2f(l_fX2_d,l_fY2_d*self.fItemHeight*l_fYRatio)
				glTexCoord1i(1)
				glVertex2f(l_fX1_d,l_fY1_d*self.fItemHeight*l_fYRatio)
				glEnd()

				glDisable(GL_TEXTURE_1D)
				glDisable(GL_BLEND)

				# Colors
				glBlendFunc(GL_DST_ALPHA,GL_ZERO)
				glEnable(GL_BLEND)
				glBegin(GL_QUADS)
				glVertex2f(l_fX1_u,l_fY1_u*self.fItemHeight*l_fYRatio)
				glVertex2f(l_fX2_u,l_fY2_u*self.fItemHeight*l_fYRatio)
				glVertex2f(l_fX2_d,l_fY2_d*self.fItemHeight*l_fYRatio)
				glVertex2f(l_fX1_d,l_fY1_d*self.fItemHeight*l_fYRatio)
				glEnd()
				glDisable(GL_BLEND)

#===============================================================================
#														Update data table
#===============================================================================
	def GetDataPointer(self,p_iIdx):
		if self.iVScaleType > 1:
			l_iValue = 'Value'
		else:
			l_iValue = self.cPlugin.lIdx[p_iIdx]
		return l_iValue

	def ResetData(self):
		self.dData			= {}
		self.cPlugin		= sta_plugin.Plugin()
		self.iNrAttrs		= 0
		self.iNrIntervals	= 0
		self.lIntervals		= []

#-------------------------------------------------------------------------------
#																	Time scale
#-------------------------------------------------------------------------------
	def UpdateTime(self):
		# Set-up the time intervals
		self.intervals = []
		self.intervals.append([])		# days
		self.intervals.append([])		# week
		self.intervals.append([])		# month
		self.intervals.append([])		# quarter
		self.intervals.append([])		# half year
		self.intervals.append([])		# year

		# Get time limits
		lStartTime = time.gmtime(sta_globals.iSelectedFilesStartTime)
		lEndTime = time.gmtime(sta_globals.iSelectedFilesEndTime)

		#-------------------------- days ---
		l_iInterval = 0
		l_cDayTime = calendar.timegm((lStartTime[0],lStartTime[1],lStartTime[2],0,0,0))
		while l_cDayTime < sta_globals.iSelectedFilesEndTime:
			self.intervals[l_iInterval].append(l_cDayTime)
			l_cDayTime = l_cDayTime + 86400
		self.intervals[l_iInterval].append(sta_globals.iSelectedFilesEndTime+1)

		#-------------------------- week ---
		l_iInterval += 1
		l_cDayTime = self.intervals[0][0]
		l_cDayTime = l_cDayTime - 86400 * lStartTime[6]
		while (l_cDayTime < sta_globals.iSelectedFilesEndTime):
			self.intervals[l_iInterval].append(l_cDayTime)
			l_cDayTime = l_cDayTime + 86400 * 7
		self.intervals[l_iInterval].append(sta_globals.iSelectedFilesEndTime+1)

		#------------------------ months ---
		l_iInterval += 1
		if (lStartTime[0] < lEndTime[0]):
			for i in range(lStartTime[1],13):
				self.intervals[l_iInterval].append(calendar.timegm((lStartTime[0],i,1,0,0,0)))
			for i in range(lStartTime[0]+1,lEndTime[0]):
				for j in range(1,13):
					self.intervals[l_iInterval].append(calendar.timegm((i,j,1,0,0,0)))
			for i in range(1,lEndTime[1]+1):
				self.intervals[l_iInterval].append(calendar.timegm((lEndTime[0],i,1,0,0,0)))
		else:
			for i in range(lStartTime[1],lEndTime[1]+1):
				self.intervals[l_iInterval].append(calendar.timegm((lStartTime[0],i,1,0,0,0)))
		self.intervals[l_iInterval].append(sta_globals.iSelectedFilesEndTime+1)

		#----------------------- quarter ---
		l_iInterval += 1
		l_iStartMonth   = (lStartTime[1]-1) / 3
		l_iEndMonth	 = (lEndTime[1]-1) / 3

		if (lStartTime[0] < lEndTime[0]):
			for i in range(l_iStartMonth,4):
				self.intervals[l_iInterval].append(calendar.timegm((lStartTime[0],i*3+1,1,0,0,0)))
			for i in range(lStartTime[0]+1,lEndTime[0]):
				for j in range(0,4):
					self.intervals[l_iInterval].append(calendar.timegm((i,j*3+1,1,0,0,0)))
			for i in range(0,l_iEndMonth+1):
				self.intervals[l_iInterval].append(calendar.timegm((lEndTime[0],i*3+1,1,0,0,0)))
		else:
			for i in range(l_iStartMonth,l_iEndMonth+1):
				self.intervals[l_iInterval].append(calendar.timegm((lStartTime[0],i*3+1,1,0,0,0)))
		self.intervals[l_iInterval].append(sta_globals.iSelectedFilesEndTime+1)

		#--------------------- half year ---
		l_iInterval += 1
		l_iStartMonth   = (lStartTime[1]-1) / 6
		l_iEndMonth	 = (lEndTime[1]-1) / 6

		if (lStartTime[0] < lEndTime[0]):
			for i in range(l_iStartMonth,2):
				self.intervals[l_iInterval].append(calendar.timegm((lStartTime[0],i*6+1,1,0,0,0)))
			for i in range(lStartTime[0]+1,lEndTime[0]):
				for j in range(0,2):
					self.intervals[l_iInterval].append(calendar.timegm((i,j*6+1,1,0,0,0)))
			for i in range(0,l_iEndMonth+1):
				self.intervals[l_iInterval].append(calendar.timegm((lEndTime[0],i*6+1,1,0,0,0)))
		else:
			for i in range(l_iStartMonth,l_iEndMonth+1):
				self.intervals[l_iInterval].append(calendar.timegm((lStartTime[0],i*6+1,1,0,0,0)))
		self.intervals[l_iInterval].append(sta_globals.iSelectedFilesEndTime+1)

		#------------------------- years ---
		l_iInterval += 1
		for i in range(lStartTime[0],lEndTime[0]+1):
			self.intervals[l_iInterval].append(calendar.timegm((i,1,1,0,0,0)))
		self.intervals[l_iInterval].append(sta_globals.iSelectedFilesEndTime+1)

#-------------------------------------------------------------------------------
#																	Data values
#-------------------------------------------------------------------------------
	def UpdateData(self):

		self.lIntervals	 = self.intervals[self.iAttrTimeInterval]		# selected sampling intervals
		self.iNrIntervals   = len(self.lIntervals)						# number of intervals

		#================================================================================
		#																Compute values
		#================================================================================
		if (self.iVScaleType > 1):

			self.iNrAttrs			= 1
			l_iValue				= 'Value'
			self.dData[l_iValue]	= [None for i in range(self.iNrIntervals)]


			# TFS has data for all commits
			if sta_globals.cSTAPrj.sType == 'TFS':
				for l_iCurrentInterval in range(self.iNrIntervals):
					l_iTime				= self.lIntervals[l_iCurrentInterval]
					l_iFiles			= 0
					l_iComputedValue	= 0

					for l_cFile in sta_globals.lSelectedFiles:
						#--- Add metric value
						if (self.cPlugin.iMetricType == 0):
							l_cCommit = None
							for l_cRev in l_cFile.lRevs:	# Find first revision before snapshot
								if (l_cRev.iTime <= l_iTime):
									l_cCommit = l_cRev
								else:
									break
							if not (l_cCommit is None):
								if (l_iTime > l_cFile.lRevs[-1].iTime) and (l_cFile.iStatus != 0):
									l_cCommit = None
							if (l_cCommit is None):
								continue

							l_iMetricValue,l_iError = self.cPlugin.getMetricValue(l_cCommit)
						else:
							l_iMetricValue,l_iError = self.cPlugin.getMetricValue(l_cFile)

						if (l_iError == 0):
							l_iComputedValue += l_iMetricValue
							l_iFiles += 1

					if (l_iFiles > 0):
						if (self.iVScaleType == 3):
							self.dData[l_iValue][l_iCurrentInterval] = 0 if l_iFiles == 0 else float(l_iComputedValue)/l_iFiles
						else:
							self.dData[l_iValue][l_iCurrentInterval] = l_iComputedValue

			# Other types than TFS only have data per snapshot
			else:
				for l_cSnapshot in sta_globals.cSTAPrj.lSnapshots:

					# Find interval
					l_iTime = l_cSnapshot[0]
					l_iCurrentInterval = 0
					while (self.lIntervals[l_iCurrentInterval] < l_iTime):
						l_iCurrentInterval += 1
					l_iCurrentInterval -= 1

					l_iFiles			= 0
					l_iComputedValue	= 0
					for l_cFile in sta_globals.lSelectedFiles:
						#--- Get snapshot revision
						l_cCommit = None
						for l_cRev in l_cFile.lRevs:	# Find first revision before snapshot
							if (l_cRev.iTime <= l_iTime):
								l_cCommit = l_cRev
							else:
								break
						if not (l_cCommit is None):
							if (l_iTime > l_cFile.lRevs[-1].iTime) and (l_cFile.iStatus != 0):
								l_cCommit = None
						if (l_cCommit is None):
							continue

						#--- Add commit value
						if (self.cPlugin.iMetricType == 0):
							l_iMetricValue,l_iError = self.cPlugin.getMetricValue(l_cCommit)
						else:
							l_iMetricValue,l_iError = self.cPlugin.getMetricValue(l_cFile)

						if (l_iError == 0):
							l_iComputedValue += l_iMetricValue
							l_iFiles += 1

					if (l_iFiles > 0):
						if (self.iVScaleType == 3):
							self.dData[l_iValue][l_iCurrentInterval] = 0 if l_iFiles == 0 else float(l_iComputedValue)/l_iFiles
						else:
							self.dData[l_iValue][l_iCurrentInterval] = l_iComputedValue

			#------------------------------------------ Propagate values ---
			l_iOldValue = 0
			for i in range(self.iNrIntervals):
				if (self.dData[l_iValue][i] is None):
					self.dData[l_iValue][i] = l_iOldValue
				else:
					l_iOldValue = self.dData[l_iValue][i]

			#------------------------------------------ Normalize values ---
			l_fMaxValue = 0
			for i in range(self.iNrIntervals):
				if self.dData[l_iValue][i] > l_fMaxValue:
					l_fMaxValue = self.dData[l_iValue][i]
			if (l_fMaxValue != 0):
				self.fDataScale = round(l_fMaxValue / self.fAttrValueFactor,3)
				for i in range(self.iNrIntervals):
					self.dData[l_iValue][i] = float(self.dData[l_iValue][i]) / l_fMaxValue
				if self.iNrIntervals>1:
					self.dData[l_iValue][-1] = self.dData[l_iValue][-2]
			else:
				self.fDataScale = 1.0
			return

		#================================================================================
		#														Count commits and files
		#================================================================================
		self.cPlugin.getSelectedIndexes()
		self.iNrAttrs	 = len(self.cPlugin.lIdx)

		#---------------------------------------------------- Compute values ---
		for i in range(self.iNrAttrs):

			l_iValue				= self.cPlugin.lIdx[i]
			self.dData[l_iValue]	= [0 for j in range(self.iNrIntervals)]

			#----------------------------------------- count commits ---
			if self.iVScaleType == 0:
				for j in sta_globals.lSelectedFiles:
					l_iCurrentInterval  = 0
					for k in j.lRevs:
						while (k.iTime > self.lIntervals[l_iCurrentInterval]):
							l_iCurrentInterval = l_iCurrentInterval + 1
						l_iCurrentInterval = l_iCurrentInterval - 1

						try:
							if (self.cPlugin.iMetricType == 0):
								l_iCompareValue = k.lValues[self.cPlugin.sClass]
							else:
								l_iCompareValue = j.lValues[self.cPlugin.sClass]
						except:
							continue

						if (l_iCompareValue == l_iValue):
							self.dData[l_iValue][l_iCurrentInterval] = self.dData[l_iValue][l_iCurrentInterval]+1

			#--------------------------- count files for categorical metrics ---
			elif (self.iVScaleType == 1) and (self.cPlugin.bHasMetricValue == False):

				for i in sta_globals.lSelectedFiles:
					l_iCurrentInterval  = 0
					j = 0
					l_iNrRevs = len(i.lRevs)
					l_bExtendedRevision = False
					l_iDecay = -1
					if (l_iNrRevs == 0):
						continue

					l_iMetricValueOld = 0
					while j<l_iNrRevs:
						#------------------------------- Get into detailed interval analysis ---
						if i.lRevs[j].iTime <= self.lIntervals[l_iCurrentInterval+1]:
							l_bHasContributed = False
							while (j<l_iNrRevs) and (i.lRevs[j].iTime <= self.lIntervals[l_iCurrentInterval+1]):

								try:
									if (self.cPlugin.iMetricType == 0):
										l_iCompareValue = i.lRevs[j].lValues[self.cPlugin.sClass]
									else:
										l_iCompareValue = i.lValues[self.cPlugin.sClass]
									l_iMetricValueOld = l_iCompareValue
								except:
									l_iCompareValue = l_iMetricValueOld

								if (l_iCompareValue == l_iValue):
									if not (l_bHasContributed):
										self.dData[l_iValue][l_iCurrentInterval] = self.dData[l_iValue][l_iCurrentInterval]+1
										l_bHasContributed = True
										l_iDecay = 0
									l_bExtendedRevision = True
								else:
									l_bExtendedRevision = False

								j = j+1

							#--- Decay
							if (not l_bHasContributed) and (self.cPlugin.bHasDecay) and (l_iDecay >=0):
								l_iDecay = l_iDecay+1
								l_iTime = self.lIntervals[l_iCurrentInterval] - self.lIntervals[l_iCurrentInterval-1]
								self.dData[l_iValue][l_iCurrentInterval] = self.dData[l_iValue][l_iCurrentInterval] + self.cPlugin.getDecayedValue(l_iDecay * l_iTime)

							l_iCurrentInterval = l_iCurrentInterval + 1
						#--------------------------- Skip to next detailed analysis interval ---
						else:

							while i.lRevs[j].iTime > self.lIntervals[l_iCurrentInterval+1]:
								#--- Decay
								if (self.cPlugin.bHasDecay) and (l_iDecay >=0):
									l_iDecay = l_iDecay+1
									l_iTime = self.lIntervals[l_iCurrentInterval] - self.lIntervals[l_iCurrentInterval-1]
									self.dData[l_iValue][l_iCurrentInterval] = self.dData[l_iValue][l_iCurrentInterval] + self.cPlugin.getDecayedValue(l_iDecay * l_iTime)
								#---------
								else:
									if (l_bExtendedRevision):
										self.dData[l_iValue][l_iCurrentInterval] = self.dData[l_iValue][l_iCurrentInterval] + 1
								l_iCurrentInterval = l_iCurrentInterval + 1

							l_bExtendedRevision = False

					#--- Fill in the remaining time if necessary
					try:
						if (self.cPlugin.iMetricType == 0):
							l_iCompareValue = i.lRevs[j-1].lValues[self.cPlugin.sClass]
						else:
							l_iCompareValue = i.lValues[self.cPlugin.sClass]
						l_iMetricValueOld = l_iCompareValue
					except:
						l_iCompareValue = l_iMetricValueOld

					if (l_iCompareValue == l_iValue):
						while l_iCurrentInterval < self.iNrIntervals:
							#--- Decay
							if (self.cPlugin.bHasDecay) and (l_iDecay >=0):
								l_iDecay = l_iDecay+1
								l_iTime = self.lIntervals[l_iCurrentInterval] - self.lIntervals[l_iCurrentInterval-1]
								self.dData[l_iValue][l_iCurrentInterval] = self.dData[l_iValue][l_iCurrentInterval] + self.cPlugin.getDecayedValue(l_iDecay * l_iTime)
							#---------
							else:
								self.dData[l_iValue][l_iCurrentInterval] = self.dData[l_iValue][l_iCurrentInterval] + 1
							l_iCurrentInterval = l_iCurrentInterval + 1

			#----------------------------- count files for numerical metrics ---
			elif (self.iVScaleType == 1) and (self.cPlugin.bHasMetricValue):

				self.dData[l_iValue] = [None for i in range(self.iNrIntervals)]

				# TFS has data for all commits
				if sta_globals.cSTAPrj.sType == 'TFS':
					for l_iCurrentInterval in range(self.iNrIntervals):
						l_iTime = self.lIntervals[l_iCurrentInterval]

						l_iFiles		= 0
						l_bValueExist	= False
						for l_cFile in sta_globals.lSelectedFiles:
							try:
								if (self.cPlugin.iMetricType == 0):
									#--- Get file revision
									l_cCommit = None
									for l_cRev in l_cFile.lRevs:	# Find first revision before interval
										if (l_cRev.iTime <= l_iTime):
											l_cCommit = l_cRev
										else:
											break
									if not (l_cCommit is None):
										if (l_iTime > l_cFile.lRevs[-1].iTime) and (l_cFile.iStatus != 0):
											l_cCommit = None
									if (l_cCommit is None):
										continue

									l_iCompareValue = l_cCommit.lValues[self.cPlugin.sClass]
								else:
									l_iCompareValue = l_cFile.lValues[self.cPlugin.sClass]
							except:
								continue

							l_bValueExist = True
							if (l_iCompareValue == l_iValue):
								l_iFiles += 1
						if (l_bValueExist):
							self.dData[l_iValue][l_iCurrentInterval] = l_iFiles

				# Other project types only have data per snapshot
				else:
					for l_cSnapshot in sta_globals.cSTAPrj.lSnapshots:

						# Find interval
						l_iTime = l_cSnapshot[0]
						l_iCurrentInterval = 0
						while (self.lIntervals[l_iCurrentInterval] < l_iTime):
							l_iCurrentInterval += 1
						l_iCurrentInterval -= 1

						l_iFiles		= 0
						l_bValueExist	= False
						for l_cFile in sta_globals.lSelectedFiles:
							#--- Get snapshot revision
							l_cCommit = None
							for l_cRev in l_cFile.lRevs:	# Find first revision before snapshot
								if (l_cRev.iTime <= l_iTime):
									l_cCommit = l_cRev
								else:
									break
							if not (l_cCommit is None):
								if (l_iTime > l_cFile.lRevs[-1].iTime) and (l_cFile.iStatus != 0):
									l_cCommit = None
							if (l_cCommit is None):
								continue

							try:
								if (self.cPlugin.iMetricType == 0):
									l_iCompareValue = l_cCommit.lValues[self.cPlugin.sClass]
								else:
									l_iCompareValue = l_cFile.lValues[self.cPlugin.sClass]
							except:
								continue

							l_bValueExist = True
							if (l_iCompareValue == l_iValue):
								l_iFiles += 1
						if (l_bValueExist):
							self.dData[l_iValue][l_iCurrentInterval] = l_iFiles

				# Propagate values
				l_iOldValue = 0
				for i in range(self.iNrIntervals):
					if (self.dData[l_iValue][i] is None):
						self.dData[l_iValue][i] = l_iOldValue
					else:
						l_iOldValue = self.dData[l_iValue][i]
		#-------------------------------------------------- Normalize values ---
		l_iMaxValue = 1
		for i in self.cPlugin.lIdx:
			for j in range(self.iNrIntervals):
				if self.dData[i][j] > l_iMaxValue:
					l_iMaxValue = self.dData[i][j]
		self.fDataScale = round(l_iMaxValue / self.fAttrValueFactor,3)
		for i in self.cPlugin.lIdx:
			for j in range(self.iNrIntervals):
				self.dData[i][j] = float(self.dData[i][j]) / l_iMaxValue
			if self.iNrIntervals>1:
				self.dData[i][-1] = self.dData[i][-2]
