#*****************************************************************************************
# File			: pextDependencies.py
# Description	: A plugin for visualizing function ID information.
# Version		: 1.0
# Copyright SolidSource B.V.
#*****************************************************************************************

#------------------------------------------------------------ Global imports ---
import wx
from OpenGL.GL import *
from OpenGL.GLUT import *
from OpenGL.GLU import *
from _winreg import *

import os
import atexit
import pysqlite2.dbapi2 as sqlite

#------------------------------------------------------------- Local imports ---
import sta_globals
import sta_plugin
import sta_utils
import sta_gl
import sta_gl_list
import sta_gl_mixerunit
import ctypes.wintypes
import win32api, win32process, win32con, win32gui, win32event
import array, struct

class COPYDATASTRUCT(ctypes.Structure):
	_fields_ = [
		('dwData', ctypes.c_void_p),
		('cbData', ctypes.c_int),
		('lpData', ctypes.c_char_p) 
	]

class SolidSX():
	def __init__(self, *args, **kwds):
		self.initSolidSXPath()
		self.sDatabasePath = None
		self.hwndSolidSX = None
		self.copyDataWindow = 0
		self.iSolution = 1
		self.reset()

		self.sMemberTable = 'D_TFS_Deps_Members'
		self.sRelationTable = 'D_TFS_Deps_Relations'

	def reset(self):
		self.solutionsDict = {}
		self.projectsDict = {}
		self.filesDict = {}
		self.namespaceDict = {}
		self.nodeDict = {}
		self.mapEdges = {}
		self.iMaxRevision = 0;

	def initSolidSXPath(self):
		self.sSolidSXPath = None
		self.sSolidSXCommand = None

		try:
			hndRegistry = ConnectRegistry(None, HKEY_CURRENT_USER)
			sKey = OpenKey(hndRegistry, r"Software\\SolidSource\\SolidSX")
			self.sSolidSXPath = QueryValue(sKey, "") + "\\bin"
			self.sSolidSXCommand = self.sSolidSXPath + "\\SolidSX2.exe" + " --listento 0 --sendto 0"  #sendto should be this windowhandler
		except WindowsError:
			pass

	def start(self):
		if self.sSolidSXPath is None:
			dial = wx.MessageDialog(None, 'SolidSX not installed!', 'Error', wx.OK | wx.ICON_ERROR)
			dial.ShowModal()
			return

		self.hwndSolidSX = win32process.CreateProcess(None, self.sSolidSXCommand, None, None, 0, 0, None, self.sSolidSXPath, win32process.STARTUPINFO())[0]
		self.initializeDatabase()

		while self.copyDataWindow is 0:
			self.copyDataWindow = win32gui.FindWindow("wxWindowClassNR", "SOLIDSX_WMCOPYDATA")
		
		self.openDatabase()

		atexit.register(self.quit)

	def quit(self):
		try:
			self.send('exit()')
			# win32api.TerminateProcess(int(self.hwndSolidSX), 0)
		except:
			pass

		self.hwndSolidSX = None
		self.copyDataWindow = 0

	def isShown(self):
		self.copyDataWindow = win32gui.FindWindow("wxWindowClassNR", "SOLIDSX_WMCOPYDATA");

		return self.copyDataWindow is not 0;

	def isRunningButUnloaded(self):
		# If the window is available but our process handle is uninitialized,
		# consider SolidSX running from an old session.
		return self.isShown() and self.hwndSolidSX is None;

	def send(self, msg):
		cds = COPYDATASTRUCT()
		cds.dwData = ctypes.c_void_p(100)
		cds.lpData = ctypes.cast(msg, c_char_p)
		cds.cbData = len(msg)
		data = ctypes.cast(ctypes.byref(cds), c_void_p)
		win32api.SendMessage(self.copyDataWindow, win32con.WM_COPYDATA, 0, cds)

	def createDatabase(self):
		self.reset();

		conn = sqlite.connect(self.sDatabasePath)
		cursor = conn.cursor()

		cursor.execute('DROP TABLE IF EXISTS edgeattr')
		cursor.execute('DROP TABLE IF EXISTS edges')
		cursor.execute('DROP TABLE IF EXISTS hierarchy')
		cursor.execute('DROP TABLE IF EXISTS information')
		cursor.execute('DROP TABLE IF EXISTS nodeattr')

		cursor.execute('CREATE TABLE edgeattr(eid INTEGER KEY, key STRING, value STRING, type STRING)')
		cursor.execute('CREATE TABLE edges(eid INTEGER PRIMARY KEY, fromnid INTEGER, tonid INTEGER)')
		cursor.execute('CREATE TABLE hierarchy(hid INTEGER, nid INTEGER, parentnid INTEGER, childorder INTEGER)')
		cursor.execute('CREATE TABLE information(key TEXT, value)')
		cursor.execute('CREATE TABLE nodeattr(nid INTEGER key, key STRING, value STRING, type STRING)')

		cursor.execute('CREATE INDEX edgeattr_eid ON edgeattr (eid)')
		cursor.execute('CREATE INDEX hierarchy_parentnid ON hierarchy (parentnid)')
		cursor.execute('CREATE INDEX hierarchy_parentnid_hid ON hierarchy (parentnid,hid)')
		cursor.execute('CREATE INDEX nodeattr_nid ON nodeattr (nid)')

		cursor.execute("insert into information (key, value) values ('information', 'SolidSX database')")
		cursor.execute("insert into information (key, value) values ('database format version', 2)")
		cursor.execute("insert into information (key, value) values ('creation date', '2014-01-01 00:00:00')")

		conn.commit()
		cursor.close()
		conn.close()

	def insertIntoHierarchy(self, j, iID, iParent, cCursorDB):
		sQuery = 'insert into hierarchy (hid, nid, parentnid, childorder) values '
		sQuery += '(' + str(j) + ', ' + str(iID) + ', ' + str(iParent) + ', 0)'
		cCursorDB.execute(sQuery)

	def insertSolution(self, sSolution, cCursorDB, iIdx):
		if self.solutionsDict.has_key(sSolution):
			return self.solutionsDict[sSolution], iIdx
		else:
			self.solutionsDict[sSolution] = iIdx
			cCursorDB.executemany("INSERT INTO nodeattr (nid, key, value) VALUES (?, ?, ?)", [
				( iIdx, 'name', sSolution ),
				( iIdx, 'type', 'Solution' ),
				( iIdx, '_icon', 'file' ),
				( iIdx, '_primary', 'True' ),
			])
			self.insertIntoHierarchy(2, iIdx, 1, cCursorDB)
			return iIdx, iIdx+1

	def insertProject(self, sProject, iSolution, cCursorDB, iIdx):
		if self.projectsDict.has_key(sProject):
			return self.projectsDict[sProject], iIdx
		else:
			self.projectsDict[sProject] = iIdx
			cCursorDB.executemany("INSERT INTO nodeattr (nid, key, value) VALUES (?, ?, ?)", [
				( iIdx, 'name', sProject ),
				( iIdx, 'type', 'Project' ),
				( iIdx, '_icon', 'file' ),
				( iIdx, '_primary', 'True' ),
			])
			self.insertIntoHierarchy(2, iIdx, iSolution, cCursorDB)
			return iIdx, iIdx+1

	def insertFile(self, iFile, cHistoryDB, cCursorDB, iIdx):
		if iFile is None:
			return None, iIdx
		elif self.filesDict.has_key(iFile):
			return self.filesDict[iFile], iIdx
		else:
			row = cHistoryDB.execute(
				"SELECT s.Path, p.Path, f.Path FROM D_TFS_Deps_Files f JOIN D_TFS_Deps_Projects p ON p.ID = f.Project " + 
				"JOIN D_TFS_Deps_Solutions s ON s.ID = f.Solution WHERE f.ID = ? LIMIT 1", (iFile,)).fetchone()

			iSolution, iIdx = self.insertSolution(row[0], cCursorDB, iIdx)
			iProject, iIdx = self.insertProject(row[1], iSolution, cCursorDB, iIdx)

			self.filesDict[iFile] = iIdx
			cCursorDB.executemany("INSERT INTO nodeattr (nid, key, value) VALUES (?, ?, ?)", [
				( iIdx, 'name', row[2] ),
				( iIdx, 'type', 'File' ),
				( iIdx, '_icon', 'classcs' ),
				( iIdx, '_primary', 'True' ),
			])
			self.insertIntoHierarchy(2, iIdx, iProject, cCursorDB)
			return iIdx, iIdx+1

	def insertNamespace(self, iFile, sName, cCursorDB, iIdx):
		lComponents = sName.split('.')
		sFQN = ""
		iParent = 1
		iFileParent = 1 if iFile is None else iFile
		for sComponent in lComponents:
			sFQN += sComponent
			if not self.namespaceDict.has_key(sFQN):
				self.namespaceDict[sFQN] = iIdx
				sQuery = 'insert into nodeattr (nid, key, value) values '
				sQuery += '(' + str(iIdx) + ", 'name', '" + sComponent + "'),"
				sQuery += '(' + str(iIdx) + ", 'type', 'Namespace'),"
				sQuery += '(' + str(iIdx) + ", '_icon', 'namespace'),"
				sQuery += '(' + str(iIdx) + ", '_primary', '" + ('False' if iFile is None else 'True') + "');"
				cCursorDB.execute(sQuery)
				self.insertIntoHierarchy(1, iIdx, iParent, cCursorDB)
				iIdx += 1
			
			iParent = iFileParent = self.namespaceDict[sFQN]
			sFQN += "."

		return self.namespaceDict[sName], iIdx

	def insertNode(self, iFile, sMember, sClass, sNamespace, iType, iParent, cCursorDB, iIdx):
		if sMember is None:
			sKey = sNamespace + "." + sClass
			sName = sClass
		else:
			sKey = sNamespace + "." + sClass + "." + sMember
			sName = sMember
	
		self.nodeDict[sKey] = iIdx
		mMemberMap = {
			# (name, icon)
			0: ('Class', 'class'),
			1: ('Struct', 'struct'),
			2: ('Enum', 'enum'),
			3: ('Interface', 'interface'),
			4: ('Constructor', 'method_public'),
			5: ('Method', 'method_public'),
			6: ('Field', 'field_public'),
		}

		sQuery = 'insert into nodeattr (nid, key, value) values '
		sQuery += '(' + str(iIdx) + ", 'name', '" + sName + "'),"
		sQuery += '(' + str(iIdx) + ", 'namespace', '" + sNamespace + "'),"
		if sMember is not None:
			sQuery += '(' + str(iIdx) + ", 'class', '" + sClass + "'),"
		sQuery += '(' + str(iIdx) + ", '_icon', '" + mMemberMap[iType][1] + "'),"
		sQuery += '(' + str(iIdx) + ", '_primary', '" + ('False' if iFile is None else 'True') + "'),"
		sQuery += '(' + str(iIdx) + ", 'type', '"+ mMemberMap[iType][0] +"');"

		cCursorDB.execute(sQuery)
		self.insertIntoHierarchy(1, iIdx, iParent, cCursorDB)

		return iIdx, iIdx+1


	def insertClass(self, lInfo, cHistoryDB, cCursorDB, iIdx):
		sNamespace = lInfo[1]
		sClass = lInfo[2]
		iType = lInfo[3]
		iFile = lInfo[4]

		iFileIdx, iIdx = self.insertFile(iFile, cHistoryDB, cCursorDB, iIdx)
		iNamespace, iIdx = self.insertNamespace(iFileIdx, sNamespace, cCursorDB, iIdx)
		iClass, iIdx = self.insertNode(iFileIdx, None, sClass, sNamespace, iType, iNamespace, cCursorDB, iIdx)

		return iClass, iIdx

	def insertMember(self, lInfo, cHistoryDB, cCursorDB, iIdx):
		sNamespace = lInfo[1]
		sClass = lInfo[2]
		sMember = lInfo[3]
		iType = lInfo[4]
		iFile = lInfo[5]

		# Possibly insert the file because the member's class may have been defined in a different file,
		# while the file this member is declared in only contains members, no class definitions
		iFileIdx, iIdx = self.insertFile(iFile, cHistoryDB, cCursorDB, iIdx)
		iClass = self.nodeDict[sNamespace+'.'+sClass]
		iMember, iIdx = self.insertNode(iFileIdx, sMember, sClass, sNamespace, iType, iClass, cCursorDB, iIdx)

		return iMember, iIdx

	def initializeNodes(self, cCursorDB, cDB, cHistoryDB, cCursorHistoryDB):
		if not sta_utils.isValidTable(cCursorHistoryDB, self.sMemberTable):
			print 'DATA: no Node information found'
			return

		cCursorDB.executemany("INSERT INTO nodeattr (nid, key, value) VALUES (?, ?, ?)", [
			( 1, 'name', 'Dataset' ),
			( 1, '_icon', 'database' ),
			( 1, '_autodisablewhenexpanded', '1' ),
			( 1, '_expand', '1' ),
			( 1, '_primary', 'False' ),
		])
		self.insertIntoHierarchy(1, 1, 'NULL', cCursorDB)
		self.insertIntoHierarchy(2, 1, 'NULL', cCursorDB)

		# Select all classes first
		sCmd = 'select ID, Namespace, Class, Type, File from '+self.sMemberTable+' where Name is null and Solution = ' + str(self.iSolution)
		iNodeIdx = 2
		cCursorHistoryDB.execute(sCmd)

		for i in cCursorHistoryDB:
			iClass, iNodeIdx = self.insertClass(i, cHistoryDB, cCursorDB, iNodeIdx)
			self.mapEdges[i[0]] = iClass

		# Now select all members
		sCmd = 'select ID, Namespace, Class, Name, Type, File from '+self.sMemberTable+' where Name is not null and Solution = ' + str(self.iSolution)
		cCursorHistoryDB.execute(sCmd)

		for i in cCursorHistoryDB:
			iMember, iNodeIdx = self.insertMember(i, cHistoryDB, cCursorDB, iNodeIdx)
			self.mapEdges[i[0]] = iMember



	def insertEdge(self, lInfo, iFrom, iTo, cCursorDB, iIdx):
		nCount = lInfo[0]
		iType = lInfo[3]
		sType = {
			# (relation name, has count)
			0: ('call', True),
			1: ('construct', True),
			2: ('access', True),
			3: ('inherits', False),
			4: ('implements', False),
			5: ('is a', False),
		}

		self.iMaxRevision = max(self.iMaxRevision, lInfo[4])

		sQueryEdges = 'insert into edges (eid, fromnid, tonid) values '
		sQueryEdges += '(' + str(iIdx) + ', ' + str(iFrom) + ', ' + str(iTo) + ')'
		sQueryEdgeAttr = 'insert into edgeattr (eid, key, value) values '
		sQueryEdgeAttr += '(' + str(iIdx) + ", 'type', '" + sType[iType][0] + "'),"
		if sType[iType][1]:
			sQueryEdgeAttr += '(' + str(iIdx) + ", 'count', " + str(nCount) + "),"
		sQueryEdgeAttr += '(' + str(iIdx) + ", 'delta', " + str(lInfo[5]) + "),"			
		sQueryEdgeAttr += '(' + str(iIdx) + ", 'revision', '" + str(lInfo[4]) + "');"
		cCursorDB.execute(sQueryEdges)
		cCursorDB.execute(sQueryEdgeAttr)

	def initializeEdges(self, cCursorDB, cDB, cHistoryDB, cCursorHistoryDB):
		if not sta_utils.isValidTable(cCursorHistoryDB, self.sRelationTable):
			print 'DATA: no Relation information found'
			return

		sCmd = 'select Count, Caller, Callee, Type, Revision, Delta from ' + self.sRelationTable + ' where Solution = ' + str(self.iSolution)
		l_iIdx = 1
		cCursorHistoryDB.execute(sCmd)
		for i in cCursorHistoryDB:
			iCaller = self.mapEdges[i[1]]
			iCallee = self.mapEdges[i[2]]
			self.insertEdge(i, iCaller, iCallee, cCursorDB, l_iIdx)
			l_iIdx = l_iIdx + 1

		return l_iIdx


	def initializeFileEdges(self, eid, cCursorDB, cDB, cHistoryDB, cCursorHistoryDB):
		if not sta_utils.isValidTable(cCursorHistoryDB, self.sRelationTable):
			print 'DATA: no Relation information found'
			return
		sCmd = 'select SUM(r.Count), r.Caller, r.Callee, r.Type, r.Revision, m1.File, m2.File from ' + self.sRelationTable + " r "
		sCmd += "join " + self.sMemberTable + " m1 on m1.ID = r.Caller "
		sCmd += "join " + self.sMemberTable + " m2 on m2.ID = r.Callee "
		sCmd += "where m2.File is not null and r.Solution = " + str(self.iSolution) + " "
		sCmd += "group by r.Caller, r.Callee, r.Type, r.Revision, m1.File, m2.File"
		l_iIdx = eid
		cCursorHistoryDB.execute(sCmd)
		for i in cCursorHistoryDB:
			iFrom = self.filesDict[i[5]]
			iTo = self.filesDict[i[6]]
			self.insertEdge(i, iFrom, iTo, cCursorDB, l_iIdx)
			l_iIdx = l_iIdx + 1

	def databasePath(self):
		return sta_globals.cSTAPrj.storagePath()+"/extra/solidsx"+str(self.iSolution)+".db"

	def initializeDatabase(self):
		self.sDatabasePath = self.databasePath()

		self.createDatabase()

		cDB = sqlite.connect(self.sDatabasePath)
		cCursorDB = cDB.cursor()

		if not sta_utils.isValidTable(cCursorDB, 'nodeattr'):
			print 'DATA: no Node Attribute information found'
			cCursorDB.close()
			cDB.close()
			return

		if not sta_utils.isValidDB(sta_globals.sDBPath):
			print 'DATA: no history database found'
			return

		cHistoryDB = sqlite.connect(sta_globals.sDBPath)

		if not sta_utils.isValidTable(cHistoryDB.cursor(), 'D_TFS_Deps_Members'):
			print 'DATA: no dependency database found'
			cHistoryDB.close()
			cDB.close()
			return

		cCursorHistoryDB = cHistoryDB.cursor()

		self.initializeNodes(cCursorDB, cDB, cHistoryDB, cCursorHistoryDB)
		eid = self.initializeEdges(cCursorDB, cDB, cHistoryDB, cCursorHistoryDB)
		self.initializeFileEdges(eid, cCursorDB, cDB, cHistoryDB, cCursorHistoryDB)

		cCursorHistoryDB.close()
		cHistoryDB.close()

		cDB.commit()
		cCursorDB.close()
		cDB.close()


	def openDatabase(self):
		if self.sDatabasePath is None or self.sDatabasePath != self.databasePath():
			self.initializeDatabase();

		self.send("ensureOpen('"+ self.sDatabasePath +"')")
		self.send("showPane('toolbar','False')")
		self.send("showPane('tree','False')")
		self.send("openView('heb','Radial View')")

		if self.namespaceDict.has_key('System'):
			iIdx = self.namespaceDict['System']
			self.send('collapseByNid(' + str(iIdx) + ', "False")')

		if self.namespaceDict.has_key(''):
			iIdx = self.namespaceDict['']
			self.send('collapseByNid(' + str(iIdx) + ', "False")')


#===============================================================================
#														Dependencies plugin
#===============================================================================
class Plugin(sta_plugin.Plugin):

	def __init__(self,*args, **kwds):
		sta_plugin.Plugin.__init__(self,*args, **kwds)
		self.sClass					= 'pextDependencies'  # Name of the plugin file
		self.sName					= 'Dependencies'	 # Name to display in a list
		self.sVersion				= '1.0'		 # Version
		self.sMetricID				= 'Dependency'

		self.solidSX = SolidSX()

#-------------------------------------------------------------------------------
#																 Filter GUI
#-------------------------------------------------------------------------------

#----------------------------------------------------------------- Build GUI ---
	def createLeftBox(self):
		leftBox = wx.BoxSizer(wx.VERTICAL)
		hBoxField = wx.BoxSizer(wx.HORIZONTAL)
		vBoxLabel = wx.BoxSizer(wx.VERTICAL)
		vBoxArea = wx.BoxSizer(wx.VERTICAL)

		fromLabel = wx.StaticText(self.tabPanel, label='From version: ')
		self.cFromField = wx.TextCtrl(self.tabPanel, style=wx.TE_RIGHT)
		self.cFromField.Bind(wx.EVT_KEY_UP, self.cbMessageFrom)

		authorLabel = wx.StaticText(self.tabPanel, label='Author: ')
		self.cFromAuthor = wx.TextCtrl(self.tabPanel, style=wx.TE_RIGHT|wx.TE_READONLY)
		self.cFromAuthor.SetValue('No author')

		infoLabel = wx.StaticText(self.tabPanel, label='Commit message: ')
		self.cInfoFromField = wx.TextCtrl(self.tabPanel, style=wx.TE_MULTILINE|wx.TE_WORDWRAP|wx.TE_READONLY)
		self.cInfoFromField.SetValue('No commit message')

		vBoxLabel.Add(fromLabel, flag=wx.TOP, border=8)
		vBoxLabel.Add(authorLabel, flag=wx.TOP, border=10)

		vBoxArea.Add(self.cFromField, flag=wx.EXPAND|wx.TOP|wx.LEFT, border=4, proportion=1)
		vBoxArea.Add(self.cFromAuthor, flag=wx.EXPAND|wx.TOP|wx.LEFT, border=4, proportion=1)

		hBoxField.Add(vBoxLabel, flag=wx.EXPAND, proportion=1)
		hBoxField.Add(vBoxArea, flag=wx.EXPAND, proportion=1)

		leftBox.Add(hBoxField, flag=wx.EXPAND|wx.TOP|wx.BOTTOM, border=10)
		leftBox.Add(infoLabel, flag=wx.BOTTOM, border=4)
		leftBox.Add(self.cInfoFromField, flag=wx.EXPAND|wx.BOTTOM, border=10, proportion=1)

		return leftBox

	def createCenterBox(self):
		centerBox = wx.BoxSizer(wx.VERTICAL)

		self.g_butShowGraph = wx.Button(self.tabPanel, -1, "Start SolidSX")
		self.g_butShowGraph.Bind(wx.EVT_BUTTON, self.cbToggleGraph)
	
		self.updateButton = wx.Button(self.tabPanel, -1, 'Update View')
		self.updateButton.Bind(wx.EVT_BUTTON, self.cbUpdate)
		self.updateButton.Disable()

		centerBox.Add(self.g_butShowGraph, flag=wx.EXPAND|wx.TOP|wx.BOTTOM, border=2)
		centerBox.Add(self.updateButton, flag=wx.EXPAND|wx.TOP|wx.BOTTOM, border=2)

		return centerBox

	def createRightBox(self):
		rightBox = wx.BoxSizer(wx.VERTICAL)
		hBoxField = wx.BoxSizer(wx.HORIZONTAL)
		vBoxLabel = wx.BoxSizer(wx.VERTICAL)
		vBoxArea = wx.BoxSizer(wx.VERTICAL)

		toLabel = wx.StaticText(self.tabPanel, label='To version: ')
		self.cToField = wx.TextCtrl(self.tabPanel, style=wx.TE_RIGHT)
		self.cToField.Bind(wx.EVT_KEY_UP, self.cbMessageTo)

		authorLabel = wx.StaticText(self.tabPanel, label='Author: ')
		self.cToAuthor = wx.TextCtrl(self.tabPanel, style=wx.TE_RIGHT|wx.TE_READONLY)
		self.cToAuthor.SetValue('No author')


		infoLabel = wx.StaticText(self.tabPanel, label='Commit message: ')
		self.cInfoToField = wx.TextCtrl(self.tabPanel, style=wx.TE_MULTILINE|wx.TE_WORDWRAP|wx.TE_READONLY)
		self.cInfoToField.SetValue('No commit message')

		vBoxLabel.Add(toLabel, flag=wx.TOP, border=8)
		vBoxLabel.Add(authorLabel, flag=wx.TOP, border=10)

		vBoxArea.Add(self.cToField, flag=wx.EXPAND|wx.TOP|wx.LEFT, border=4)
		vBoxArea.Add(self.cToAuthor, flag=wx.EXPAND|wx.TOP|wx.LEFT, border=4)
		
		hBoxField.Add(vBoxLabel, flag=wx.EXPAND, proportion=1)
		hBoxField.Add(vBoxArea, flag=wx.EXPAND, proportion=1)

		rightBox.Add(hBoxField, flag=wx.EXPAND|wx.TOP|wx.BOTTOM, border=10)
		rightBox.Add(infoLabel,  flag=wx.BOTTOM, border=4)
		rightBox.Add(self.cInfoToField, flag=wx.EXPAND|wx.BOTTOM, border=10, proportion=1)

		return rightBox

	def createBottomBox(self):
		box = wx.BoxSizer(wx.HORIZONTAL)

		label = wx.StaticText(self.tabPanel, label='Solution: ')

		self.solutionsDropdown = wx.ComboBox(self.tabPanel, -1, choices=[], style=wx.CB_DROPDOWN|wx.CB_READONLY)
		self.solutionsDropdown.Bind(wx.EVT_COMBOBOX, self.cbUpdateSolution)

		box.Add(label, flag=wx.BOTTOM|wx.LEFT|wx.RIGHT, border=10)
		box.Add(self.solutionsDropdown, flag=wx.EXPAND|wx.BOTTOM|wx.LEFT|wx.RIGHT, border=10, proportion=1)

		return box

	def cbUpdateSolution(self, event):
		iIdx = self.solutionsDropdown.GetSelection()
		self.solidSX.iSolution = self.solutions[iIdx][0]

		self.solidSX.openDatabase()


	def buildFilterGUI(self,pMasterGUI):

		self.cGUIMaster = pMasterGUI
		mainFrame = sta_globals.GUI.mainFrame;

		self.tabPanel = wx.Panel(mainFrame.notebook_1, -1)
		mainFrame.notebook_1.AddPage(self.tabPanel, "Dependencies")

		sizer = wx.BoxSizer(wx.VERTICAL)

		self.tabPanel.SetSizer(sizer)
		sizer.Fit(self.tabPanel)
		sizer.SetSizeHints(self.tabPanel)

		combineBox = wx.BoxSizer(wx.HORIZONTAL)

		leftBox = self.createLeftBox()
		centerBox = self.createCenterBox()
		rightBox = self.createRightBox()
		bottomBox = self.createBottomBox()

		combineBox.Add(leftBox, flag=wx.EXPAND|wx.LEFT|wx.RIGHT, border=10, proportion=1)
		combineBox.Add(centerBox, flag=wx.TOP|wx.BOTTOM|wx.LEFT|wx.RIGHT, border=10)
		combineBox.Add(rightBox, flag=wx.EXPAND|wx.LEFT|wx.RIGHT, border=10, proportion=1)

		sizer.Add(combineBox, flag=wx.EXPAND|wx.LEFT|wx.RIGHT|wx.BOTTOM, border=0, proportion=1)
		sizer.Add(bottomBox, flag=wx.EXPAND|wx.LEFT|wx.RIGHT|wx.BOTTOM, border=0, proportion=0)

		self.tabPanel.Enable(False)

		self.cGUIList = sta_gl_list.wxColorCheckListBox(self.sName,self.sClass,self.tabPanel, style=wx.BORDER_NONE|wx.LB_SORT|wx.TRANSPARENT_WINDOW)
		self.cGUIList.Hide()

	def enableFilter_aux(self,l_sw):
		self.tabPanel.Enable(l_sw)
		sta_plugin.Plugin.enableFilter_aux(self, False)

#-------------------------------------------------------------------------------
#														 Filter GUI callback
#-------------------------------------------------------------------------------

	def updateMessage(self, sValue, commitField, authorField):
		if not sta_utils.isValidDB(sta_globals.sDBPath):
			print 'DATA: no history database found'
			return

		cHistoryDB = sqlite.connect(sta_globals.sDBPath)
		cCursorHistoryDB = cHistoryDB.cursor()
		
		sQueryCommit = "select m.Name from M_SS2007010102 m join Versions v ON m.ID = v.ID where v.Name = ? limit 1;"
		cCursorHistoryDB.execute(sQueryCommit, (sValue,))

		commitField.SetValue("No commit message")
		for m in cCursorHistoryDB:
			commitField.SetValue(str(m[0]))

		sQueryAuthor = "select m.Name from M_SS2007010101 m join Versions v ON m.ID = v.ID where v.Name = ? limit 1;"
		cCursorHistoryDB.execute(sQueryAuthor, (sValue,))

		authorField.SetValue("No author")
		for m in cCursorHistoryDB:
			authorField.SetValue(str(m[0]))


	def cbMessageFrom(self, event):
		self.parseVersions()

		sValue = self.cFromField.GetValue()
		self.updateMessage(sValue, self.cInfoFromField, self.cFromAuthor)

	def cbMessageTo(self, event):
		self.parseVersions()

		sValue = self.cToField.GetValue()
		self.updateMessage(sValue, self.cInfoToField, self.cToAuthor)


	def cbToggleGraph(self, event):
		if self.solidSX.isRunningButUnloaded():
			self.solidSX.openDatabase();
			self.solidSX.hwndSolidSX = 0;
		 	self.g_butShowGraph.SetLabel("Quit SolidSX")
		 	self.updateButton.Enable()
		elif not self.solidSX.isShown():
			self.solidSX.start()
		 	self.g_butShowGraph.SetLabel("Quit SolidSX")
		 	self.updateButton.Enable()
		else:
			self.solidSX.quit()
			self.g_butShowGraph.SetLabel("Start SolidSX")
		 	self.updateButton.Disable()

	def nodeFiler(self, lMembers):
		iID = lMembers[0]
		sNamespace = lMembers[1]
		sClass = lMembers[2]
		sMember = lMembers[3]

		if sMember is None:
			sKey = sNamespace + "." + sClass
		else:
			sKey = sNamespace + "." + sClass + "." + sMember
		iMemberIdx = self.solidSX.nodeDict[sKey]

		return '("id (*)", ' + str(iMemberIdx) + ')'

	def parseVersions(self):
		sFrom = str(self.cFromField.GetValue())
		sTo = str(self.cToField.GetValue())

		try:
			iFrom = int(sFrom)
		except ValueError:
			iFrom = None

		try:
			iTo = int(sTo)
		except ValueError:
			iTo = None

		self.cFromField.SetValue('' if iFrom is None else str(iFrom))
		self.cToField.SetValue('' if iTo is None else str(iTo))

		return iFrom, iTo

	def cbUpdate(self, event):
		if not sta_utils.isValidDB(sta_globals.sDBPath):
			print 'DATA: no history database found'
			return

		cHistoryDB = sqlite.connect(sta_globals.sDBPath)
		cCursorHistoryDB = cHistoryDB.cursor()

		iFrom, iTo = self.parseVersions()

		if iFrom is None and iTo is None:
			# self.solidSX.send("setNodeInclusionFilter(None)")
			# self.solidSX.send("setEdgeInclusionFilter(None)")
			self.solidSX.send("setEdgeColorAttribute('type')")
			self.solidSX.send("setEdgeColorMap('Rainbow')")

			return

		sQuery = "SELECT DISTINCT m.ID, m.Namespace, m.Class, m.Name, m.File FROM D_TFS_Deps_Members m "
		sQuery += "INNER JOIN D_TFS_Deps_Relations r ON (m.ID = r.Caller OR m.ID = r.Callee) "
		sQuery += "WHERE m.Solution = ? "
		if iFrom is not None and iTo is not None:
			sQuery += "AND r.Revision >= ? AND r.Revision <= ?"
			bindings = (self.solidSX.iSolution, iFrom, iTo)
		elif iFrom is not None:
			sQuery += "AND r.Revision >= ?"
			bindings = (self.solidSX.iSolution, iFrom)
		elif iTo is not None:
			sQuery += "AND r.Revision <= ?"
			bindings = (self.solidSX.iSolution, iTo)

		cCursorHistoryDB.execute(sQuery, bindings)

		lFilterNodes = []
		lFilterEdges = []
		for m in cCursorHistoryDB:
			lFilterNodes.append(self.nodeFiler(m))

			if m[4] is not None:
				iID = self.solidSX.filesDict[m[4]]
				lFilterNodes.append('("id (*)", ' + str(iID) + ')')

		l_iFrom = 0 if iFrom is None else iFrom
		l_iTo = self.solidSX.iMaxRevision if iTo is None else iTo

		for r in xrange(l_iFrom, l_iTo + 1):
			lFilterEdges.append('("revision", ' + str(r) + ')')

		self.solidSX.send("setNodeInclusionFilter([" + ', '.join(lFilterNodes) + "])")
		self.solidSX.send("setEdgeInclusionFilter([" + ', '.join(lFilterEdges) + "])")
		self.solidSX.send("setEdgeColorAttribute('delta')")
		self.solidSX.send("setEdgeColorMap('GreenYellowRed')")

	def showFilterGUI(self,p_bState):
		self.loadMetric()
		pass

	def loadMetric(self):
		if not sta_utils.isValidDB(sta_globals.sDBPath):
			print 'DATA: no dependency database found'
			return

		cHistoryDB = sqlite.connect(sta_globals.sDBPath)

		if not sta_utils.isValidTable(cHistoryDB.cursor(), 'D_TFS_Deps_Solutions'):
			print 'DATA: no dependency database found'
			cHistoryDB.close()
			return

		self.solutions = []
		self.solutionsDropdown.Clear()

		for s in cHistoryDB.execute("select ID, Path from D_TFS_Deps_Solutions"):
			self.solutions.append(s)
			self.solutionsDropdown.Append(s[1])

		if len(self.solutions) > 0:
			self.solutionsDropdown.SetSelection(0)
			self.solidSX.iSolution = self.solutions[0][0]

		cHistoryDB.close()

	def unloadMetric(self):
		self.solidSX.quit();
		pass

#===============================================================================
#											Dependencies filter mixer unit
#===============================================================================

class MixerUnit(sta_gl_mixerunit.MixerUnit):

	def __init__(self, *args, **kwds):
		sta_gl_mixerunit.MixerUnit.__init__(self, *args, **kwds)

		self.iAttrType		= 0
		self.iDrawType		= 0
