#*****************************************************************************************
# Name			: Folder ID
# File			: pextFolder.py
# Description	: A plugin for visualizing folder ID information.
# Version		: 1.0
# Copyright SolidSource B.V.
#*****************************************************************************************

#------------------------------------------------------------ Global imports ---
import wx
from OpenGL.GL import *
from OpenGL.GLUT import *
from OpenGL.GLU import *

import math
import base64
import pysqlite2.dbapi2 as sqlite

#------------------------------------------------------------- Local imports ---
import sta_globals
import sta_plugin
import sta_utils
import sta_gl_list
import sta_gl_mixerunit

#===============================================================================
#													Configuration dialog box
#===============================================================================

class CSettingsDlg(wx.Dialog):
	def __init__(self, pParent, *args, **kwds):

		self.pParent = pParent

		kwds["style"] = wx.DEFAULT_DIALOG_STYLE

		wx.Dialog.__init__(self, *args, **kwds)
		self.txt1 = wx.StaticText(self, -1, "   Look-up period")
		self.txt2 = wx.StaticText(self, -1, "   File number relevance")
		self.txt3 = wx.StaticText(self, -1, "   Activity threshold")
		self.txt4 = wx.StaticText(self, -1, "   TRisk reference")
		self.txt5 = wx.StaticText(self, -1, "   Activity reference")
		self.txt6 = wx.StaticText(self, -1, "   LRisk reference")

		self.comboTimeRef	= wx.ComboBox(self, -1, choices=['6 months','12 months','18 months','2 years','3 years','4 years','5 years','10 years'], style=wx.CB_DROPDOWN|wx.CB_READONLY)
		self.comboFileRel	= wx.ComboBox(self, -1, choices=['Very low','Low','Medium','High','Very high'], style=wx.CB_DROPDOWN|wx.CB_READONLY)
		self.spinActivity	= wx.SpinCtrl(self, -1, '')
		self.spinActivity.SetRange(1,10)
		self.txtRiskRef		= wx.TextCtrl(self, -1, '')
		self.txtLRiskRef	= wx.TextCtrl(self, -1, '')
		self.txtActivityRef	= wx.TextCtrl(self, -1, '')
		self.bOK			= wx.Button(self, -1, "Ok")
		self.bCancel		= wx.Button(self, -1, "Cancel")

		self.__set_properties()
		self.__do_layout()
		self.__get_values()

		self.bOK.Bind(wx.EVT_BUTTON,self.OnOK)
		self.bCancel.Bind(wx.EVT_BUTTON,self.OnCancel)

	def __set_properties(self):
		self.SetTitle('Folder metric settings')
		self.comboTimeRef.SetMinSize((150, -1))
		self.comboFileRel.SetMinSize((150, -1))
		self.spinActivity.SetMinSize((50, -1))
		self.txtRiskRef.SetMinSize((150, -1))
		self.txtLRiskRef.SetMinSize((150, -1))
		self.txtActivityRef.SetMinSize((150,-1))
		self.bOK.SetMinSize((50, -1))
		self.bCancel.SetMinSize((50, -1))

	def __do_layout(self):

		self.sizerMain = wx.GridSizer(6,2,0,0)
		self.sizerMain.Add(self.txt1, 0, wx.ALIGN_CENTER_VERTICAL|wx.ADJUST_MINSIZE, 0)
		self.sizerMain.Add(self.comboTimeRef, 0, wx.ALIGN_CENTER_VERTICAL|wx.ADJUST_MINSIZE, 0)
		self.sizerMain.Add(self.txt2, 0, wx.ALIGN_CENTER_VERTICAL|wx.ADJUST_MINSIZE, 0)
		self.sizerMain.Add(self.comboFileRel, 0, wx.ALIGN_CENTER_VERTICAL|wx.ADJUST_MINSIZE, 0)
		self.sizerMain.Add(self.txt3, 0, wx.ALIGN_CENTER_VERTICAL|wx.ADJUST_MINSIZE, 0)
		self.sizerMain.Add(self.spinActivity, 0, wx.ALIGN_CENTER_VERTICAL|wx.ADJUST_MINSIZE, 0)
		self.sizerMain.Add(self.txt4, 0, wx.ALIGN_CENTER_VERTICAL|wx.ADJUST_MINSIZE, 0)
		self.sizerMain.Add(self.txtRiskRef, 0, wx.ALIGN_CENTER_VERTICAL|wx.ADJUST_MINSIZE, 0)
		self.sizerMain.Add(self.txt6, 0, wx.ALIGN_CENTER_VERTICAL|wx.ADJUST_MINSIZE, 0)
		self.sizerMain.Add(self.txtLRiskRef, 0, wx.ALIGN_CENTER_VERTICAL|wx.ADJUST_MINSIZE, 0)
		self.sizerMain.Add(self.txt5, 0, wx.ALIGN_CENTER_VERTICAL|wx.ADJUST_MINSIZE, 0)
		self.sizerMain.Add(self.txtActivityRef, 0, wx.ALIGN_CENTER_VERTICAL|wx.ADJUST_MINSIZE, 0)
		self.sizerMain.Add(self.bOK, 0, wx.ALL|wx.ALIGN_RIGHT|wx.ADJUST_MINSIZE, 10)
		self.sizerMain.Add(self.bCancel, 0, wx.ALL|wx.ALIGN_LEFT|wx.ADJUST_MINSIZE, 10)

		self.SetAutoLayout(True)
		self.SetSizer(self.sizerMain)
		self.sizerMain.Fit(self)
		self.sizerMain.SetSizeHints(self)
		self.Layout()

	def __get_values(self):
		self.comboTimeRef.SetSelection(self.pParent.iPastRef_GUI)
		self.comboFileRel.SetSelection(self.pParent.iFileRelevence_GUI)
		self.spinActivity.SetValue(self.pParent.iCommitCountRef)
		self.txtRiskRef.SetValue(str(self.pParent.fRiskRef))
		self.txtLRiskRef.SetValue(str(self.pParent.fLRiskRef))
		self.txtActivityRef.SetValue(str(self.pParent.fActivityRef))

	def OnOK(self, event):
		try:

			self.pParent.fRiskRef			= int(self.txtRiskRef.GetValue())

			self.pParent.fLRiskRef			= int(self.txtLRiskRef.GetValue())

			self.pParent.fActivityRef		= int(self.txtActivityRef.GetValue())

			self.pParent.iPastRef_GUI 		= self.comboTimeRef.GetSelection()
			self.pParent.iPastRef			= self.pParent.iStepRef * self.pParent.iPastRefList[self.pParent.iPastRef_GUI]

			self.pParent.iFileRelevence_GUI	= self.comboFileRel.GetSelection()
			self.pParent.iFileRelevence		= self.pParent.iFileRelevenceList[self.pParent.iFileRelevence_GUI]

			self.pParent.iCommitCountRef	= self.spinActivity.GetValue()

			self.pParent.cGUIList.SetMetricReference('Activity',self.pParent.fActivityRef)
			self.pParent.cGUIList.SetMetricReference('TRisk index',self.pParent.fRiskRef)
			self.pParent.cGUIList.SetMetricReference('LRisk index',self.pParent.fLRiskRef)

			self.pParent.cGUIList.RefreshMetric()

			self.EndModal(wx.ID_OK)
			event.Skip()
		except:
			dlg = wx.MessageDialog(sta_globals.GUI.mainFrame,'Reference values should be integer','Incorrect value',style=wx.OK|wx.ICON_EXCLAMATION)
			dlg.ShowModal()
			dlg.Destroy()

	def OnCancel(self, event):
		self.EndModal(wx.ID_CANCEL)
		event.Skip()


#===============================================================================
#															  Extension plugin
#===============================================================================
class Plugin(sta_plugin.Plugin):

	def __init__(self,*args, **kwds):
		sta_plugin.Plugin.__init__(self,*args, **kwds)
		self.sClass					= 'pextFolder'  # Name of the plugin file
		self.sName					= 'Folder'	  # Name to display in a list
		self.iMetricType			= 1			 # File plugin
		self.sVersion				= '1.0'		 # Version
		self.sMetricID				= 'FLD'

		self.sTable					= 'Files'		# Name of the table where metric is stored
		self.lTables.append(self.sTable)				# Append to this list all tables that are used

		#----------------------------------------------------- Specific processing

		self.iLevel					= 1			  # Current folder nesting level
		self.iMaxLevel				= 1			  # Maximum folder nesting level

		self.lPaths					= []			 # Full folder paths
		self.lLevelPaths			= []			 # Level nesting size paths
		self.dPathMap				= {}			 # Map between levels

		self.iStepRef				= 15768000			# Reference step
		self.iPastRef				= self.iStepRef * 2	# Past reference to look for activity (1 year)
		self.iPastRefList			= [1,2,3,4,6,8,10,20]
		self.iPastRef_GUI			= 1
		self.iCommitCountRef		= 1					# From how many commits we consider activity taking place
		self.iFileRelevenceList		= [0,0.25,0.5,0.75,1.0]
		self.iFileRelevence			= 0.5				# What is the influence of the number of files on the team risk
		self.iFileRelevence_GUI		= 2
		self.fRiskRef				= 10				# Experimental maximum team risk reference value
		self.fLRiskRef				= 30				# Experimental maximum legacy risk reference value
		self.fActivityRef			= 100				# Experimental maximum activity reference value

		self.lSrcCode				= ['java',',jsp','js','py','pl','sql','idl','xsl']
		self.lNonCode				= ['html','htm','css','xml','txt','pdf','gif','jpg','bmp','class']

#-------------------------------------------------------------------------------
#																 Filter GUI
#-------------------------------------------------------------------------------


#----------------------------------------------------------------- Build GUI ---
	def buildFilterGUI(self,pMasterGUI):

		self.cGUIMaster = pMasterGUI

		self.cSizer = wx.BoxSizer(wx.VERTICAL)
		l_cConfigSizer = wx.BoxSizer(wx.HORIZONTAL)

		l_cLabelSizer = wx.BoxSizer(wx.HORIZONTAL)
		l_sLabel = wx.StaticText(pMasterGUI.project_filters, -1, '  '+self.sName)
		self.g_txtMetric = wx.StaticText(pMasterGUI.project_filters, -1, style = wx.ALIGN_RIGHT )
		l_cLabelSizer.Add(l_sLabel, 0, wx.ADJUST_MINSIZE, 0)
		l_cLabelSizer.Add(self.g_txtMetric, 0, wx.ALIGN_RIGHT, 0)

		self.cGUIList = sta_gl_list.wxColorCheckListBox(self.sName,self.sClass,pMasterGUI.project_filters, style=wx.BORDER_SUNKEN|wx.LB_MULTIPLE|wx.LB_SORT)
		self.cGUIList.lObservers.append(self)

		self.cGUIList.SetData(sta_globals.lIDColors)
		self.cGUIList.cMenu.Enable(212,False)  # Marker center
		self.cGUIList.cMenu.Enable(213,False)  # Marker density

		self.cGUIScale = wx.Slider(pMasterGUI.project_filters, -1, self.iLevel, 1, 20)
		self.cGUIConfig = wx.Button(pMasterGUI.project_filters, -1, 'cfg')
		self.cGUIScale.SetOwnBackgroundColour(wx.Colour(150,150,180))

		self.cGUIList.AddMetric("# files",self.MetricFolderFileNumber)
		self.cGUIList.AddMetric("Activity",self.MetricFolderActivity,self.fActivityRef)
		self.cGUIList.AddMetric("Text hits",self.MetricFolderTextHits)
		self.cGUIList.AddMetric("LRisk index",self.MetricLegacyRisk,self.fLRiskRef)
		self.cGUIList.AddMetric("TRisk index",self.MetricTeamRisk,self.fRiskRef)

		self.cGUIList.Bind(wx.EVT_LISTBOX,self.cbListSelect)
		self.cGUIScale.Bind(wx.EVT_COMMAND_SCROLL_THUMBRELEASE,self.cbLevelSelect)
		self.cGUIScale.Bind(wx.EVT_COMMAND_SCROLL_PAGEUP,self.cbLevelSelect)
		self.cGUIScale.Bind(wx.EVT_COMMAND_SCROLL_PAGEDOWN,self.cbLevelSelect)
		self.cGUIConfig.Bind(wx.EVT_BUTTON,self.cbConfig)

		self.cGUIList.SetMinSize((240,-1))
		self.cGUIConfig.SetMinSize((30,15))
		self.cGUIScale.SetMinSize((-1,15))

		self.cSizer.Add(l_cLabelSizer, 0, wx.EXPAND, 0)
		self.cSizer.Add(self.cGUIList, 1, wx.EXPAND|wx.ADJUST_MINSIZE, 0)
		l_cConfigSizer.Add(self.cGUIScale, 1, wx.EXPAND, 0)
		l_cConfigSizer.Add(self.cGUIConfig, 0, wx.ADJUST_MINSIZE, 0)
		self.cSizer.Add(l_cConfigSizer, 0, wx.EXPAND, 0)

		pMasterGUI.cListsSizer_Project.Add(self.cSizer, 0, wx.EXPAND, 0)

		self.showFilterGUI(False)
		self.doBindings()

	def cbConfig(self,evt):
		dlg = CSettingsDlg(self,sta_globals.GUI.mainFrame)
		val = dlg.ShowModal()
		dlg.Destroy()

#-------------------------------------------------------------------------------
#								 Filter GUI callback
#-------------------------------------------------------------------------------

	def cbListSelect_aux(self, files):
		if len(self.lIdx) == 0:
			for i in files:
				if i.iType == 0:
					i.lValues[self.sClass] = self.dPathMap[i.lValues[self.sMetricID]]
				else:
					i.lValues[self.sClass] = -1
		else:
			for i in files:
				if i.iType == 0:
					l_iIdx = self.dPathMap[i.lValues[self.sMetricID]]
					i.lValues[self.sClass] = -1
					if l_iIdx in self.lIdx:
						i.lValues[self.sClass] = l_iIdx
				else:
					i.lValues[self.sClass] = -1

	def getSelectedIndexes(self):
		l_lsFileT =  self.cGUIList.GetSelections()
		self.clearSelection()
		for i in l_lsFileT:
			self.lIdx.append(self.lLevelPaths.index(i))
		self.bSelected = len(self.lIdx)>0

	def cbLevelSelect(self,evt):

		if (self.iLevel != self.cGUIScale.GetValue()):

			sta_globals.GUI.startTimelyAction()

			self.iLevel = self.cGUIScale.GetValue()

			self.lData  = []
			self.lIdx   = []
			self.clearFilterGUI()

			self.lLevelPaths = []
			self.dPathMap = {}
			l_iIdx = 0

			for i in range(len(self.lPaths)):
				l_sPath = self.lPaths[i]
				l_iLevel = l_sPath.count('/')+1

				if self.iLevel < l_iLevel:
					l_iPos = -1
					for j in range(self.iLevel):
						l_iPos = l_sPath.find('/',l_iPos+1)
					if l_iPos > 0:
						l_sPath = l_sPath[:l_iPos]
					else:
						l_sPath = '.'

				if (not l_sPath in self.lLevelPaths):
					self.lLevelPaths.append(l_sPath)
					self.cGUIList.Append(l_sPath,l_iIdx)
					self.lData.append(l_iIdx)
					self.dPathMap[i] = l_iIdx
					l_iIdx = l_iIdx+1
				else:
					self.dPathMap[i] = self.lLevelPaths.index(l_sPath)

			#self.cbListSelect(0)
			for i in sta_globals.lFiles:
				if i.iType == 0:
					i.lValues[self.sClass] = self.dPathMap[i.lValues[self.sMetricID]]
			self.cGUIList.MenuSortAlphabetically(0)
			sta_globals.GUI.glCanvas.Refresh()
			sta_globals.GUI.glMixerCanvas_Project.Refresh()

			sta_globals.GUI.stopTimelyAction()

#-------------------------------------------------------------------------------
#									Filter GUI metrics
#-------------------------------------------------------------------------------
#---------------------------------------------- Number of files in file type ---
	def MetricFolderFileNumber(self,p_sName):
		iResult = 0
		l_iIdx = self.lLevelPaths.index(p_sName)
		for i in sta_globals.lSelectedFiles:
			if (self.dPathMap[i.lValues[self.sMetricID]] == l_iIdx):
				iResult = iResult + 1

		return (iResult)

#------------------------------------------------------------ Activity index ---
	def MetricFolderActivity(self,p_sName):
		iResult = 0
		iFiles  = 0
		iTimeReference	= sta_globals.iProjectEndTime - self.iPastRef

		l_iStartTime = sta_globals.iStartFocusInterval
		if (l_iStartTime < 0):
			l_iStartTime = sta_globals.iSelectedFilesStartTime

		l_iEndTime = sta_globals.iEndFocusInterval
		if (l_iEndTime < 0):
			l_iEndTime = sta_globals.iSelectedFilesEndTime

		l_iIdx = self.lLevelPaths.index(p_sName)
		for i in sta_globals.lSelectedFiles:
			if (self.dPathMap[i.lValues[self.sMetricID]] == l_iIdx):

				fFactorActivity	= 0
				for j in i.lRevs:

					if (j.iTime >= l_iStartTime) and (j.iTime <= l_iEndTime):
						fFactorActivity = fFactorActivity + 1

				iResult = iResult + fFactorActivity
				iFiles = iFiles + 1

		if (iFiles != 0):
			return (float(iResult)/math.pow(iFiles,self.iFileRelevence))
		else:
			return 0
#------------------------------------------------------------ Text hits index ---
	def MetricFolderTextHits(self,p_sName):
		iResult = 0
		iFiles  = 0

		l_iStartTime = sta_globals.iStartFocusInterval
		if (l_iStartTime < 0):
			l_iStartTime = sta_globals.iSelectedFilesStartTime

		l_iEndTime = sta_globals.iEndFocusInterval
		if (l_iEndTime < 0):
			l_iEndTime = sta_globals.iSelectedFilesEndTime

		l_iIdx = self.lLevelPaths.index(p_sName)
		for i in sta_globals.lSelectedFiles:
			if (self.dPathMap[i.lValues[self.sMetricID]] == l_iIdx):

				fFactorActivity	= 0
				for j in i.lRevs:
					try:
						if (j.iTime >= l_iStartTime) and (j.iTime <= l_iEndTime) and (j.lValues['pextFind'] != -1):
							fFactorActivity = fFactorActivity + 1
					except:
						pass

				iResult = iResult + fFactorActivity
				iFiles = iFiles + 1

		if (iFiles != 0):
			return iResult
		else:
			return 0
#----------------------------------------------------------- Team risk index ---
	def MetricTeamRisk(self,p_sName):
		fResult = 0
		iFiles 	= 0
		iTimeReference	= sta_globals.iProjectEndTime - self.iPastRef

		l_iIdx = self.lLevelPaths.index(p_sName)
		for i in sta_globals.lSelectedFiles:
			if (self.dPathMap[i.lValues[self.sMetricID]] == l_iIdx):

				fFactorActivity	= 0
				lDevelopers 	= set([])
				for j in i.lRevs:
					if j.iTime > iTimeReference:
						lDevelopers.add(sta_globals.GUI.auxGetAuthor(j))
						fFactorActivity = fFactorActivity + 1
				fFactorActivity = math.sqrt(fFactorActivity)

				if fFactorActivity >= self.iCommitCountRef:

					iFiles = iFiles + 1
					fFactorDevelopers = math.pow(len(lDevelopers),4)
					fContribution = fFactorActivity/fFactorDevelopers
					fResult = fResult + fContribution

		if (fResult != 0):
			fResult = fResult/math.pow(iFiles,self.iFileRelevence)

		return (fResult)
#-------------------------------------------------------- Legacy risk index ---
	def MetricLegacyRisk(self,p_sName):
		fResult = 0
		iFiles 	= 0
		iTimeReference	= sta_globals.iProjectEndTime
		iYearInterval	= 31536000							# Number of seconds in one year

		l_iIdx = self.lLevelPaths.index(p_sName)
		for i in sta_globals.lSelectedFiles:
			if (self.dPathMap[i.lValues[self.sMetricID]] == l_iIdx):

				iPos = i.sPath.rfind('.')
				sExt = i.sPath[iPos+1:]
				if (sExt not in self.lNonCode) and (len(i.lRevs)>0):
					fResult = fResult + float(iTimeReference - i.lRevs[-1].iTime)/iYearInterval
					iFiles = iFiles + 1

		if (fResult != 0):
			fResult = fResult/math.pow(iFiles,self.iFileRelevence)

		return (fResult)
#-------------------------------------------------------------------------------
#																Data management
#-------------------------------------------------------------------------------
#--------------------------------------------------------------- Load metric ---
	def loadMetric(self):

		#--------------------------------- Build list of folders ---
		self.lPaths = []
		for i in sta_globals.lFiles:
			if i.iType == 0:
				l_sPath = i.sPath
				l_iLevel = l_sPath.count('/')
				l_iIdx = l_sPath.rfind('/')
				if self.iMaxLevel < l_iLevel:
					self.iMaxLevel = l_iLevel

				if l_iIdx > 0:
					l_sPath = l_sPath[:l_iIdx]
				else:
					l_sPath = '.'
				if not l_sPath in self.lPaths:
					self.lPaths.append(l_sPath)
				iClass = self.lPaths.index(l_sPath)

				i.lValues[self.sMetricID] = iClass
			else:
				i.lValues[self.sMetricID] = -1
				i.lValues[self.sClass] = -1

		self.cGUIScale.SetRange(1,self.iMaxLevel)
		self.cGUIScale.SetValue(min(2,self.iMaxLevel))
		self.iLevel = -1

		#--------------------------------------- Build GUI list ---
		self.cbLevelSelect(0)
		#---------------------------------------- Assign values ---
		self.clearSelection()


#------------------------------------------------------------- Unload metric ---
	def unloadMetric(self):
		self.iMaxLevel = 1

		sta_plugin.Plugin.unloadMetric(self)

#===============================================================================
#						Extension filter mixer unit
#===============================================================================
class MixerUnit(sta_gl_mixerunit.MixerUnit):

	def __init__(self, *args, **kwds):
		sta_gl_mixerunit.MixerUnit.__init__(self, *args, **kwds)

		self.iAttrType		= 1
		self.iDrawType 		= 2
