﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using ManyConsole;
using Ionic.Zip;
using CSharpMetrics;
using CSharpMetrics.Scopes;

namespace SolidTA.TFSImporter.Commands
{
    class CalculateMetrics : TfsCommand
    {
        protected Analyzer Analyzer = new Analyzer();

        public CalculateMetrics()
        {
            IsCommand("metrics", "Calculates simple metrics of C# files");
        }

        public override int Run(string[] remainingArguments)
        {
            var inputArguments = ParseInputArguments();

            InterceptSignals();

            MayExit = false;

            OpenDatabase();
            CreateTablesIfNecessary();

            ExitIfCancelled();
            Execute(remainingArguments, inputArguments);

            return (int)ExitCode.Success;
        }

        protected override void Execute(string[] args, string[] ids)
        {
            // Do *not* initialize the progress bar, its length is already set and progress for
            // non C# files has already been reported, so we should not reset the progress bar.
            // InitProgress(ids.Length, 2);

            Database.BeginTransaction();

            foreach (var id in ids)
            {
                ExitIfCancelled();

                try
                {
                    CalculateFileMetrics(id);
                }
                catch (System.IO.FileNotFoundException)
                {
                    var file = Database.Find<Models.File>(id);

                    Console.WriteLine("No file contents downloaded for {0}", file != null ? file.Path : "unknown file");
                }
                catch (System.IO.DirectoryNotFoundException)
                {
                    Console.WriteLine("No file contents downloaded yet.");

                    break;
                }

                UpdateProgress(2);
            }

            Database.Commit();
        }

        protected void CalculateFileMetrics(string id)
        {
            var path = ArchivePathForFile(id);

            using (var zipFile = ZipFile.Read(path))
            {
                var versions = Database.Query<Models.Version>(@"
                    select V.* from Versions V
                    left join M_SS2007010201 M on M.ID = V.ID
                    where V.File = ? and M.ID is null", id);

                foreach (var version in versions)
                {
                    ExitIfCancelled();

                    if (zipFile.ContainsEntry(version.Name))
                    {
                        CalculateVersionMetrics(version, zipFile);
                    }
                }
            }
        }

        protected void CalculateVersionMetrics(Models.Version version, ZipFile zipFile)
        {
            using (var source = new System.IO.MemoryStream())
            {
                zipFile[version.Name].Extract(source);

                source.Seek(0, System.IO.SeekOrigin.Begin);

                File file = Analyzer.Analyze(null, source);

                InsertAnalyzeResults(version, file);
            }
        }

        protected void InsertAnalyzeResults(Models.Version version, File file)
        {
            Database.Insert(new Models.Size.File
            {
                ID = version.ID,
                MOD = file.NrClasses,
                MTD = file.NrMethods,
                LOC = file.LOC,
                MCB = file.Complexity,
                INV = file.Invocations,
                COM = file.CLOC,
                IDEP = file.NrImports,
                IF4 = 0,
                IF4c = 0,
                IF4v = 0,
            });

            InsertFileClasses(version, file);
        }

        protected void InsertFileClasses(Models.Version version, File file)
        {
            foreach (var ns in file.Namespaces.Values)
            {
                foreach (var klass in ns.Classes.Values)
                {
                    Database.Insert(new Models.Size.Class
                    {
                        ID = version.ID,
                        Module = klass.FullName,
                        LOC = klass.LOC,
                        MTD = klass.NrMethods,
                        PROP = klass.NrFields,
                        MCB = klass.Complexity,
                        INV = klass.Invocations,
                        COM = klass.CLOC,
                        WMC1 = 0,
                        WMCv = 0,
                        DIT = 0,
                        NOC = 0,
                        CBO = 0,
                        FIv = 0,
                        FIc = 0,
                        FI = 0,
                        FOv = 0,
                        FOc = 0,
                        FO = 0,
                        IF4v = 0,
                        IF4c = 0,
                        IF4 = 0,
                    });

                    InsertClassMethods(version, klass);
                }
            }
        }

        protected void InsertClassMethods(Models.Version version, Class klass)
        {
            foreach (var method in klass.Methods)
            {
                Database.Insert(new Models.Size.Method
                {
                    ID = version.ID,
                    Module = klass.FullName,
                    Prototype = method.Prototype,
                    SSIZE = method.NrParams,
                    LOC = method.LOC,
                    MCB = method.Complexity,
                    COM = method.CLOC,
                    INV = method.Invocations,
                    STC = method.MaxDepth,
                });
            }
        }

        protected string ArchivePathForFile(string id)
        {
            return string.Format("./extra/files/{0}.zip", id);
        }

        protected override void CreateTablesIfNecessary()
        {
            base.CreateTablesIfNecessary();

            if (Database.CreateTable<Models.Size.File>())
            {
                Database.Insert(new Models.Metric
                {
                    ID = "M_SS2007010201",
                    Name = "Basic metrics",
                    Type = Models.Metric.Types.Version,
                    Description = "Basic code metrics.<br />Computed using the basic metrics calculator.",
                    Dependencies = "D_SS2007010201_1:D_SS2007010201_2",
                });
            }

            Database.CreateTable<Models.Size.Class>();

            Database.CreateTable<Models.Size.Method>();
        }
    }
}
