﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using CSharpDeps.Scopes;
using SolidTA.TFSImporter.Collections;

namespace SolidTA.TFSImporter.Commands.Deps
{
    namespace State
    {
        class File
        {
            public List<int> VersionIDs = new List<int>();
            public HashSet<Class> Classes = new HashSet<Class>();
            public HashSet<Method> Methods = new HashSet<Method>();

            public int FanIn
            {
                get { return Classes.Select(c => c.FanIn).Sum(); }
            }

            public int FanOut
            {
                get { return Classes.Select(c => c.FanOut).Sum(); }
            }

            public float Cohesion { get; set; }
        }

        class Class
        {
            public string Name;
            public HashSet<string> In = new HashSet<string>();
            public HashSet<string> Out = new HashSet<string>();

            public int FanIn
            {
                get { return In.Count(); }
            }

            public int FanOut
            {
                get { return Out.Count(); }
            }

            public float Cohesion { get; set; }
        }

        class Method
        {
            public string ClassName;
            public string Name;
            public HashSet<string> In = new HashSet<string>();
            public HashSet<string> Out = new HashSet<string>();

            public int FanIn
            {
                get { return In.Count(); }
            }

            public int FanOut
            {
                get { return Out.Count(); }
            }

            public float Cohesion { get; set; }
        }
    }

    class MetricsProcessor : Processor
    {
        protected static int? PreviousVersion;

        protected State.File FileVersion;

        protected Dictionary<string, State.Class> Classes = new Dictionary<string, State.Class>();

        protected Dictionary<string, State.Method> Methods = new Dictionary<string, State.Method>();

        protected Dictionary<int, State.File> Files = new Dictionary<int, State.File>();

        public override void PreProcess()
        {
            var from = PreviousVersion.HasValue ? PreviousVersion + 1 : Changeset.ChangesetId;
            var versions = Database.Query<Models.Version>(@"
                SELECT * FROM Versions
                WHERE CAST(Name AS INTEGER) BETWEEN ? AND ?",
                from, Changeset.ChangesetId);
            
            // Build file map
            foreach (var version in versions)
            {
                State.File file;

                if (!Files.TryGetValue(version.File, out file))
                {
                    Files[version.File] = file = new State.File();
                }

                file.VersionIDs.Add(version.ID);
            }
        }

        public override void PostProcess()
        {
            RemoveOwnReferences();

            SaveVersionedResults();
            SaveClassResults();
            SaveMethodResults();

            PreviousVersion = Changeset.ChangesetId;
        }

        protected void RemoveOwnReferences()
        {
            // Remove references to ourselves
            foreach (var klass in Classes.Values)
            {
                klass.In.Remove(klass.Name);
                klass.Out.Remove(klass.Name);
            }

            // Remove references to ourselves
            foreach (var method in Methods.Values)
            {
                method.In.Remove(method.ClassName);
                method.Out.Remove(method.ClassName);
            }
        }

        protected void SaveVersionedResults()
        {
            foreach (var file in Files.Values)
            {
                foreach (var version in file.VersionIDs)
                {
                    foreach (var klass in file.Classes)
                    {
                        Database.Insert(new Models.Deps.VersionedClass
                        {
                            ID = version,
                            FI = klass.FanOut,
                            FO = klass.FanOut,
                            Cohesion = klass.Cohesion,
                        });
                    }

                    foreach (var method in file.Methods)
                    {
                        Database.Insert(new Models.Deps.VersionedMethod
                        {
                            ID = version,
                            FI = method.FanOut,
                            FO = method.FanOut,
                            Cohesion = method.Cohesion,
                        });
                    }
                }
            }
        }

        protected void SaveClassResults()
        {
            foreach (var klass in Classes.Values)
            {
                Database.Insert(new Models.Deps.ClassMetric
                {
                    Version = Changeset.ChangesetId,
                    Class = klass.Name,
                    FI = klass.FanIn,
                    FO = klass.FanOut,
                    Cohesion = klass.Cohesion,
                });
            }
        }

        protected void SaveMethodResults()
        {
            foreach (var method in Methods.Values)
            {
                Database.Insert(new Models.Deps.MethodMetric
                {
                    Version = Changeset.ChangesetId,
                    Method = method.Name,
                    FI = method.FanIn,
                    FO = method.FanOut,
                    Cohesion = method.Cohesion,
                });
            }
        }

        public override void Process(Solution solution, Project project, File file)
        {
            // Find the version object this file belongs to
            if (File.TFSFile.HasValue)
            {
                Files.TryGetValue(File.TFSFile.Value, out FileVersion);
            }
            
            base.Process(solution, project, file);
        }

        public override void Process(File file, Namespace ns, Class klass)
        {
            var c = GetClass(klass.FullName);
            var nrPairs = 0;

            // Consider class properties for fan-out metric and calculate cohesion
            foreach (var property in klass.Properties)
            {
                c.Out.Add(property.Type.FullName);

                // For each method in this class, count the number of methods accessing this property.
                var accessedFrom = 0;
                var fullName = string.Format("{0}.{1}", klass.FullName, property.Name);

                foreach (var method in klass.Methods)
                {
                    foreach (var access in method.MemberAccesses.Values)
                    {
                        if (access.FullName == fullName) accessedFrom++;
                    }
                }

                // From the number of methods this property is accessed from, calculate the number
                // of pairs according to n!/(k!*(n-k)!) for k=2 which is n*(n-1)/2
                nrPairs += accessedFrom * (accessedFrom - 1) / 2;
            }

            // The class cohesion can now be calculated from the number of pairs divided by the total number of methods
            c.Cohesion = klass.Methods.Count() > 0 ? nrPairs / klass.Methods.Count() : -1.0f;

            base.Process(file, ns, klass);
        }

        public override void Process(File file, Namespace ns, Class klass, Method method)
        {
            var fullName = klass.FullName;

            var c = GetClass(fullName);
            var m = GetMethod(fullName, method.FullName);

            // Link the class and method with the current version
            if (FileVersion != null)
            {
                FileVersion.Classes.Add(c);
                FileVersion.Methods.Add(m);
            }

            // Include function calls in fan-in/out
            foreach (var invocation in method.Invocations.Values)
            {
                string className = invocation.Class.FullName;

                c.Out.Add(className);
                m.Out.Add(className);

                GetClass(className).In.Add(fullName);
                GetMethod(className, invocation.FullName).In.Add(fullName);
            }

            var accessedProperties = new HashSet<string>();

            // Include property accesses in fan-in/out
            foreach (var access in method.MemberAccesses.Values)
            {
                string className = access.Class.FullName;

                c.Out.Add(className);
                m.Out.Add(className);

                GetClass(className).In.Add(fullName);

                // Collect internal property accesses
                if (access.FullName == fullName)
                {
                    accessedProperties.Add(access.Name);
                }
            }

            // Calculate method cohesion as number of internal accesses vs. number of used classes
            m.Cohesion = m.Out.Count() > 0 ? accessedProperties.Count() / m.Out.Count() : -1.0f;
        }

        protected State.Class GetClass(string name)
        {
            State.Class klass;

            if (Classes.TryGetValue(name, out klass))
            {
                return klass;
            }
            else
            {
                return Classes[name] = new State.Class { Name = name };
            }
        }

        protected State.Method GetMethod(string klass, string name)
        {
            State.Method method;

            if (Methods.TryGetValue(name, out method))
            {
                return method;
            }
            else
            {
                return Methods[name] = new State.Method { ClassName = klass, Name = name };
            }
        }
    }
}
