﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using CSharpDeps.Scopes;
using SolidTA.TFSImporter.Collections;

namespace SolidTA.TFSImporter.Commands.Deps
{
    using Member = Models.Deps.Member;
    using Relation = Models.Deps.Relation;

    class RelationsProcessor : Processor
    {
        protected static bool Initialized = false;

        public class SolutionState
        {
            public bool Processed = false;

            public TwoKeyDictionary<int, int, Relation> Relations
                = new TwoKeyDictionary<int, int, Relation>();

            public Dictionary<int, Member> ID2Member
                = new Dictionary<int, Member>();

            public Dictionary<string, Member> Name2Member
                = new Dictionary<string, Member>();
        }

        protected static Dictionary<int, SolutionState> Solutions
            = new Dictionary<int, SolutionState>();

        protected SolutionState State;

        public override void PreProcess()
        {
            // Only load the relations and members once
            if (!Initialized)
            {
                MapMembers();
                MapRelations();

                Initialized = true;
            }
        }

        protected void MapMembers()
        {
            var members = Database.Table<Member>();

            foreach (var member in members)
            {
                SolutionState state = GetState(member.Solution);

                state.Name2Member[member.FullName] = member;
                state.ID2Member[member.ID] = member;
            }
        }

        protected void MapRelations()
        {
            // Query the most recent relations
            var relations = Database.Query<Relation>(@"
                SELECT r.* FROM D_TFS_Deps_Relations r
                INNER JOIN (
                    SELECT *, MAX(ID) AS MaxID
                    FROM D_TFS_Deps_Relations
                    GROUP BY Caller, Callee)
                AS j ON r.ID = j.MaxID
                WHERE r.Count <> 0"
            );

            foreach (var relation in relations)
            {
                GetState(relation.Solution).Relations[relation.Caller, relation.Callee] = relation;
            }
        }

        public override void PostProcess()
        {
            foreach (var solution in Solutions.Values.Where(s => s.Processed))
            {
                // Reset processed flag
                solution.Processed = false;

                // Iterate over all relations and negate the ones which have not been touched,
                // e.g. these relations have not been found again and should be removed.
                foreach (var relation in solution.Relations.Map.Values.Where(r => !r.Touched && r.Count > 0).ToArray())
                {
                    InsertRelation(new Relation
                    {
                        Delta = -relation.Count,
                        Count = 0,
                        Caller = relation.Caller,
                        Callee = relation.Callee,
                        Type = relation.Type,
                        Solution = relation.Solution,
                        Project = relation.Project,
                    });
                }
            }
        }

        protected SolutionState GetState(int solutionID)
        {
            SolutionState state;

            if (Solutions.TryGetValue(solutionID, out state))
            {
                return state;
            }
            
            return Solutions[solutionID] = new SolutionState();
        }

        public void PrepareForSolution(Models.Deps.Solution solution)
        {
            State = GetState(solution.ID);

            // Mark the solution as processed
            State.Processed = true;
        }

        public void PrepareForProject(Models.Deps.Project project)
        {
            // For each known relation known for this project, mark it as not having been touched
            foreach (var relation in State.Relations.Map.Values.Where(r => r.Project == project.ID))
            {
                relation.Touched = false;
            }
        }

        public override void Process(File file, Namespace ns, Class klass)
        {
            var node = FindMember(klass.FullName, () => new Member
            {
                File = File.ID,
                Namespace = ns.FullName,
                Class = klass.Name,
                Type = ClassType(klass.Type),
            });

            UpdateMemberFile(node);

            // Save inherits relation
            if (klass.Inherits != null)
            {
                var inherits = FindClass(klass.Inherits);

                EnsureRelation(node, inherits, Relation.Types.Inherits);
            }

            // Save implements relations
            foreach (var implements in klass.Implements)
            {
                var interfaceNode = FindClass(implements);

                EnsureRelation(node, interfaceNode, Relation.Types.Implements);
            }

            // Insert class properties
            foreach (var property in klass.Properties)
            {
                var fullName = string.Format("{0}.{1}", klass.FullName, property.Name);

                var member = FindMember(fullName, () => new Member
                {
                    File = File.ID,
                    Namespace = ns.FullName,
                    Class = klass.Name,
                    Name = property.Name,
                    Type = Member.Types.Property,
                });

                UpdateMemberFile(member);

                // Get property type member
                var isNode = FindClass(property.Type);

                EnsureRelation(member, isNode, Relation.Types.IsA);
            }

            base.Process(file, ns, klass);
        }

        public override void Process(File file, Namespace ns, Class klass, Method method)
        {
            var caller = FindMember(method.FullName, () => new Member
            {
                File = File.ID,
                Namespace = ns.FullName,
                Class = klass.Name,
                Name = method.Name,
                Type = MethodType(method.Name),
            });

            UpdateMemberFile(caller);

            // Store invocations
            foreach (var invocation in method.Invocations.Values)
            {
                // Make sure the invoked class is present in the database
                var invokedClass = FindClass(invocation.Class);

                var callee = FindMember(invocation.FullName, () => new Member
                {
                    Namespace = invocation.Class.Namespace,
                    Class = invocation.Class.Name,
                    Name = invocation.Name,
                    Type = MethodType(invocation.Name),
                });

                var delta = CalculateDelta(caller, callee, invocation.Invocations);

                if (delta != 0)
                {
                    InsertRelation(new Relation
                    {
                        Delta = delta,
                        Count = invocation.Invocations,
                        Caller = caller.ID,
                        Callee = callee.ID,
                        Type = (callee.Type == Member.Types.Constructor)
                            ? Relation.Types.Construct
                            : Relation.Types.Call,
                    });
                }
            }

            // Store member accesses
            foreach (var access in method.MemberAccesses.Values)
            {
                // Make sure the accessed class is present in the database
                var accessedClass = FindClass(access.Class);

                var member = FindMember(access.FullName, () => new Member
                {
                    Namespace = access.Class.Namespace,
                    Class = access.Class.Name,
                    Name = access.Name,
                    Type = Member.Types.Property,
                });

                var delta = CalculateDelta(caller, member, access.Accesses);

                if (delta != 0)
                {
                    InsertRelation(new Relation
                    {
                        Delta = delta,
                        Count = access.Accesses,
                        Caller = caller.ID,
                        Callee = member.ID,
                        Type = Relation.Types.Access,
                    });
                }
            }
        }

        protected int CalculateDelta(Member from, Member to, int count)
        {
            var relation = FindRelation(from, to);

            return relation != null ? count - relation.Count : count;
        }

        protected static Member.Types MethodType(string name)
        {
            if (name.StartsWith(".ctor"))
            {
                return Member.Types.Constructor;
            }
            else
            {
                return Member.Types.Method;
            }
        }

        protected static Member.Types ClassType(Class.Types type)
        {
            switch (type)
            {
                case Class.Types.Class:
                    return Member.Types.Class;
                case Class.Types.Enum:
                    return Member.Types.Enum;
                case Class.Types.Interface:
                    return Member.Types.Interface;
                case Class.Types.Struct:
                    return Member.Types.Struct;
            }

            return Member.Types.Class;
        }

        protected void UpdateMemberFile(Member member)
        {
            // Update the member's file to the file it currently is in
            if (member.File != File.ID)
            {
                member.File = File.ID;

                Database.Update(member);
            }
        }

        protected Member FindClass(ClassReference klass)
        {
            return FindMember(klass.FullName, () => new Member
            {
                Namespace = klass.Namespace,
                Class = klass.Name,
                Type = ClassType(klass.Type),
            });
        }

        protected Member FindMember(string name, Func<Member> creator)
        {
            Member member;

            if (!State.Name2Member.TryGetValue(name, out member))
            {
                member = creator();
                member.Solution = Solution.ID;

                Try(() => Database.Insert(member));

                State.Name2Member[name] = member;
                State.ID2Member[member.ID] = member;
            }

            return member;
        }

        protected Relation FindRelation(Member from, Member to)
        {
            if (State.Relations.ContainsKey(from.ID, to.ID))
            {
                var relation = State.Relations[from.ID, to.ID];

                // Touch the relation as we have seen it now
                relation.Touched = true;

                return relation;
            }
            else
            {
                return null;
            }
        }

        protected void EnsureRelation(Member from, Member to, Relation.Types type)
        {
            // If the relation is not currently present, insert it now
            if (FindRelation(from, to) == null)
            {
                InsertRelation(new Relation
                {
                    Delta = 1,
                    Count = 1,
                    Caller = from.ID,
                    Callee = to.ID,
                    Type = type,
                });
            }
        }

        protected void InsertRelation(Relation relation)
        {
            relation.Revision = Changeset.ChangesetId;
            relation.Touched = true;

            if (relation.Solution == 0)
            {
                relation.Solution = Solution.ID;
                relation.Project = Project.ID;
            }

            Database.Insert(relation);

            GetState(relation.Solution).Relations[relation.Caller, relation.Callee] = relation;
        }
    }
}
