﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.IO;
using Ionic.Zip;
using Microsoft.TeamFoundation.VersionControl.Client;

namespace SolidTA.TFSImporter.Commands
{
    class ImportFileContents : TfsCommand
    {
        public ImportFileContents()
        {
            IsCommand("contents", "Imports the contents of the specified files at the given snapshot times");

            RequiresAdditionalArgumentsViaInput();
        }
        
        protected override void Execute(string[] args, string[] ids)
        {
            InitProgress(ids.Length);

            foreach (var id in ids)
            {
                ExitIfCancelled();

                var file = Database.Find<Models.File>(id);

                if (file != null)
                {
                    try
                    {
                        ImportVersions(file);
                    }
                    catch (ChangesetNotFoundException)
                    {
                        // File is already up-to-date
                    }
                }
                else
                {
                    Console.WriteLine("File with id [{0}] does not exist in the database.", id);
                }

                UpdateProgress();
            }
        }

        protected void ImportVersions(Models.File file)
        {
            var path = ArchivePathForFile(file);

            using (var zipFile = new ZipFile(path))
            {
                var begin = LatestImportedChangelog(zipFile) + 1;

                var changesets = VersionControlServer.QueryHistory(file.Path, VersionSpec.Latest, 0, RecursionType.None,
                    null, new ChangesetVersionSpec(begin), VersionSpec.Latest, int.MaxValue, true, true, true, true);

                foreach (Changeset changeset in changesets)
                {
                    ExitIfCancelled();
                    
                    PutChangesetContents(changeset, zipFile);
                }

                zipFile.Save();
            }
        }

        protected void PutChangesetContents(Changeset changeset, ZipFile zipFile)
        {
            var item = changeset.Changes[0].Item;

            using (Stream data = GetFileContents(item))
            {
                if (data != null)
                {
                    zipFile.AddEntry(changeset.ChangesetId.ToString(), GetBytes(data));
                }
            }
        }

        protected Stream GetFileContents(Item item)
        {
            try
            {
                ExitIfCancelled();

                return item.DownloadFile();
            }
            catch (VersionControlException e)
            {
                Console.WriteLine(e.Message);

                return null;
            }
        }

        protected byte[] GetBytes(Stream input)
        {
            using (var output = new MemoryStream())
            {
                byte[] buffer = new byte[32768];
                int read;

                while ((read = input.Read(buffer, 0, buffer.Length)) > 0)
                {
                    output.Write(buffer, 0, read);
                }

                return output.ToArray();
            }
        }

        protected int LatestImportedChangelog(ZipFile zipFile)
        {
            return zipFile.EntryFileNames.Count > 0 ? zipFile.EntryFileNames.Select(n => int.Parse(n)).Max() : 0;
        }

        protected string ArchivePathForFile(Models.File file)
        {
            return string.Format("./extra/files/{0}.zip", file.ID);
        }
    }
}
