﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using ManyConsole;
using Microsoft.TeamFoundation.VersionControl.Client;
using Microsoft.TeamFoundation.WorkItemTracking.Client;

namespace SolidTA.TFSImporter.Commands
{
    class ImportFileVersions : TfsCommand
    {
        public ImportFileVersions()
        {
            IsCommand("versions", "Imports all version history of the specified files into the database.");

            RequiresAdditionalArgumentsViaInput();
        }

        protected override void Execute(string[] args, string[] ids)
        {
            InitProgress(ids.Length);

            Database.BeginTransaction();

            foreach (var id in ids)
            {
                ExitIfCancelled();

                var file = Database.Find<Models.File>(id);

                if (file != null)
                {
                    try
                    {
                        ImportVersions(file);
                    }
                    catch (ChangesetNotFoundException)
                    {
                        // File is already up-to-date
                    }
                }
                else
                {
                    Console.WriteLine("File with id [{0}] does not exist in the database.", id);
                }

                UpdateProgress();
            }

            Database.Commit();
        }

        protected void ImportVersions(Models.File file)
        {
            int begin = LatestImportedChangelog(file) + 1;

            var changesets = VersionControlServer.QueryHistory(file.Path, VersionSpec.Latest, 0, RecursionType.Full,
                null, new ChangesetVersionSpec(begin), VersionSpec.Latest, int.MaxValue, true, true, false, true);

            foreach (Changeset changeset in changesets)
            {
                ExitIfCancelled();

                var version = new Models.Version
                {
                    File = file.ID,
                    Name = changeset.ChangesetId.ToString(),
                    Time = changeset.CreationDate,
                };

                Database.Insert(version);

                Database.Insert(new Models.Metrics.Author
                {
                    ID = version.ID,
                    Name = changeset.Committer,
                });

                Database.Insert(new Models.Metrics.Comment
                {
                    ID = version.ID,
                    Name = changeset.Comment,
                });
            }
        }

        protected int LatestImportedChangelog(Models.File file)
        {
            return Database.Table<Models.Version>().Where(v => v.File == file.ID).Max("Name") ?? 0;
        }
    }
}
