﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using ManyConsole;
using SQLite;
using SolidTA.TFSImporter.Exceptions;
using Microsoft.TeamFoundation;
using Microsoft.TeamFoundation.VersionControl.Client;

namespace SolidTA.TFSImporter.Commands
{
    abstract class TfsCommand : ConsoleCommand
    {
        public string Url { get; private set; }

        public string Path { get; private set; }

        public string User { get; private set; }

        public string Password { get; private set; }

        public bool AdditionalArgumentsViaInput { get; set; }

        public bool MayExit { get; set; }

        public bool Cancelled { get; set; }

        public bool Progress { get; set; }

        public SQLiteConnection Database { get; private set; }

        private HandlerRoutine SignalHandler;

        public TFSServer Server = TFSServerFactory.Create();

        protected VersionControlServer VersionControlServer
        {
            get { return Server.GetService<VersionControlServer>(); }
        }

        public TfsCommand()
        {
            HasRequiredOption("U|url=", "The URL of the TFS server.", v => Url = v);
            HasOption("P|path=", "Optional path relative to the root of the repository.", v => Path = v);
            HasOption("u|user=", "TFS user to authenticate with.", v => User = v);
            HasOption("p|password=", "TFS password for the user.", v => Password = v);
            HasOption("d|directory=", "Use as current working directory.", v => Environment.CurrentDirectory = v);
            HasOption("progress", "Output progress information", v => Progress = v != null);

            SkipsCommandSummaryBeforeRunning();
        }

        protected string[] ParseInputArguments()
        {
            string line;
            var ids = new List<string>();

            while ((line = Console.ReadLine()) != null && !line.Equals("."))
            {
                ids.Add(line);
            }

            return ids.ToArray();
        }

        public override int Run(string[] remainingArguments)
        {
            var inputArguments = AdditionalArgumentsViaInput ? ParseInputArguments() : new string[] { };

            MayExit = true;
            InterceptSignals();

            Path = PreparePath(Path);

            Console.WriteLine("Connecting to TFS...");
            ExitIfCancelled();
            Connect();

            MayExit = false;

            ExitIfCancelled();
            OpenDatabase();

            ExitIfCancelled();
            CreateTablesIfNecessary();

            Console.WriteLine("Connected, executing request...");
            ExitIfCancelled();
            Execute(remainingArguments, inputArguments);

            return (int)ExitCode.Success;
        }

        protected abstract void Execute(string[] remainingArguments, string[] inputArguments);

        protected void RequiresAdditionalArgumentsViaInput()
        {
            AdditionalArgumentsViaInput = true;
        }

        protected void InitProgress(int total, int gauge = 0)
        {
            if (Progress)
            {
                Console.WriteLine("##{0}#{1}", gauge, total);
            }
        }

        protected void UpdateProgress(int gauge = 0)
        {
            if (Progress)
            {
                Console.WriteLine("#{0}", gauge);
            }
        }

        protected void ExitIfCancelled(bool commit = false)
        {
            if (Cancelled)
            {
                if (commit) Database.Commit();

                // Throw an exception here so any resources will be properly disposed off.
                throw new CancelledException();
            }
        }

        protected void InterceptSignals()
        {
            // Storing the handler in an ivar is necessary to prevent the GC from disposing it.
            SignalHandler = new HandlerRoutine(HandleSignal);

            Program.SetConsoleCtrlHandler(SignalHandler, true);
        }

        private bool HandleSignal(CtrlTypes signal)
        {
            switch (signal)
            {
                case CtrlTypes.CTRL_C_EVENT:
                case CtrlTypes.CTRL_BREAK_EVENT:
                    if (MayExit)
                    {
                        Environment.Exit((int)ExitCode.Cancelled);
                    }

                    Console.WriteLine("Process cancelled, may take a while before process has stopped.");

                    Cancelled = true;
                    break;
            }

            return true;
        }

        protected string PreparePath(string path)
        {
            if (string.IsNullOrEmpty(Path))
            {
                return "$/";
            }
            else if (path.StartsWith("$/"))
            {
                return path;
            }
            else if (path.StartsWith("/"))
            {
                return "$" + path;
            }
            else if (path.StartsWith("./"))
            {
                return "$" + path.Substring(1);
            }
            else
            {
                return "$/" + path;
            }
        }

        protected void Connect()
        {
            try
            {
                Server.Connect(Url, User, Password);
            }
            catch (ArgumentException e)
            {
                throw new AbortException(ExitCode.ConnectionFailed, e);
            }
            catch (TeamFoundationServerUnauthorizedException e)
            {
                throw new AbortException(ExitCode.AuthenticationFailed, e);
            }
        }

        protected void OpenDatabase()
        {
            Database = new SQLiteConnection("./history.db", DateTimeStorage.Unix);

            Database.EnableForeignKeys();
        }

        protected virtual void CreateTablesIfNecessary()
        {
            Database.CreateTable<Models.Version>();
            Database.CreateTable<Models.File>();
            Database.CreateTable<Models.Metric>();

            if (Database.CreateTable<Models.Metrics.Author>())
            {
                Database.Insert(new Models.Metric
                {
                    ID = "M_SS2007010101",
                    Name = "Authors",
                    Type = Models.Metric.Types.Version,
                    Description = "Developer identity",
                    Dependencies = null,
                });
            }

            if (Database.CreateTable<Models.Metrics.Comment>())
            {
                Database.Insert(new Models.Metric
                {
                    ID = "M_SS2007010102",
                    Name = "Comments",
                    Type = Models.Metric.Types.Version,
                    Description = "Version associated message log",
                    Dependencies = null,
                });
            }
        }
    }
}
