#include <sstream>
#include <stdio.h>
#include "VoxelFile.h"

using namespace std;

const int VoxelFile::BINVOX = 0;
const int VoxelFile::VT = 1;
const int VoxelFile::MIRA = 2;
const int VoxelFile::RAWVOX = 3;
const int VoxelFile::VTK = 4;

static const int NR_EXTENSIONS = 5;
const string VoxelFile::extensions[] = {"binvox", "vt", "mira", "raw", "vtk"};

const string VoxelFile::DEFAULT_VTK_COMMENT_STRING =
  "generated by [binvox], http://www.google.com/search?q=binvox";





VoxelFile::VoxelFile(Voxels& voxels_ref, string filename) :
  VoxelRef(voxels_ref),
  my_filename(filename)
{
  input = 0;
  output = 0;

  with_types = 1;
  
  vtk_comment_string = DEFAULT_VTK_COMMENT_STRING;
  
}  // constructor



VoxelFile::~VoxelFile()
{
  close();
  
}  // destructor



void
VoxelFile::set_filespec(bool saving)
{
  //  cout << "VoxelFile::set_filespec" << endl;

  if (my_filetype < NR_EXTENSIONS) {
    //
    // check if this file already exists (only if saving)
    //
    bool filename_ok = !saving;

    if (filename_ok) {
      total_filespec = total_filespec + my_filename + "." + extensions[my_filetype];
      cout << "total_filespec [" << total_filespec << "]" << endl;
    }
    else {
      string test_filespec;
      int counter = 0;
      while(!filename_ok) {
	test_filespec = total_filespec;
	test_filespec += my_filename;
	if (counter) {
	  test_filespec += "_";
	  char counter_string[8];
	  sprintf(counter_string, "%d", counter);
	  test_filespec += counter_string;
	}
	test_filespec += ".";
	test_filespec += extensions[my_filetype];
	if (counter) cout << "trying [" << test_filespec << "]" << endl;
      
	fstream inp_test;
	inp_test.open(test_filespec.c_str(), ios::in);
	if (inp_test.is_open()) {
	  inp_test.close();
	  cout << "  [" << test_filespec << "] already exists" << endl;
	  counter++;
	}
	else
	  filename_ok = true;
      }  // while
    
      total_filespec = test_filespec;
    }  // else, filename not ok
  }
  else
    cout << "VoxelFile::set_filespec(" << my_filetype << ") error: unknown file type" << endl;
  
}  // VoxelFile::set_filespec



int
VoxelFile::open_for_read(int type)
{
  my_filetype = type;
  set_filespec(false);
  
#if IRIX
  input = new ifstream(total_filespec.c_str(), ios::in);
#else
  input = new ifstream(total_filespec.c_str(), ios::in | ios::binary);
#endif
  
  if (!input->good()) {
    cout << "Error opening [" << total_filespec << "]" << endl;
    delete input;
    input = 0;
    return 0;
  }

  return 1;
  
}  // VoxelFile::open_for_read



int
VoxelFile::open_for_write(int type)
{
  my_filetype = type;
  set_filespec(true);
  
#if IRIX
  output = new ofstream(total_filespec.c_str(), ios::out);
#else
  output = new ofstream(total_filespec.c_str(), ios::out | ios::binary);
#endif
  
  if (!output->good()) {
    cout << "Error creating [" << total_filespec << "]" << endl;
    delete output;
    output = 0;
    return 0;
  }

  return 1;
  
}  // VoxelFile::open_for_write



void
VoxelFile::close()
{
  if (input) {
    input->close();
    input = 0;
  }
  if (output) {
    output->close();
    output = 0;
  }

}  // VoxelFile::close



int
VoxelFile::load(string filespec)
{
  string::size_type dot_pos = filespec.rfind('.');
  string extension = filespec.substr(dot_pos + 1);
  filespec.erase(dot_pos);
  
  cout << "  filename: [" << filespec << "], extension: [" << extension << "]" << endl;

  return load(filespec, extension);
  
}  // VoxelFile::load



int
VoxelFile::get_filetype(string extension)
{
  //  cout << "VoxelFile::get_filetype(" << extension << ")" << endl;
  
  int filetype = -1;
  for(int i=0; i < extension.length(); i++) extension[i] |= 0x20;  // lowercase it first
  
  for(int i=0; (i < NR_EXTENSIONS) && (filetype == -1); i++) {
    if (extension.compare(extensions[i]) == 0) filetype = i;
  }
  
  return filetype;
  
}  // VoxelFile::get_filetype



int
VoxelFile::load(string filename, string extension)
{
  my_filetype = get_filetype(extension);
  
  switch(my_filetype) {
    case BINVOX: return read_binvox();
    case VT: return read_vt();
    case MIRA: return read_mira();
    default:
      cout << "  error: unsupported filetype " << my_filetype << endl;
  }  // switch

  return 0;
  
}  // VoxelFile::load



int
VoxelFile::read_binvox()
{
  //
  // read header
  //
  string line;
  *input >> line;  // #binvox
  if (line.compare("#binvox") != 0) {
    cout << "Error: first line reads [" << line << "] instead of [#binvox]" << endl;
    delete input;
    return 0;
  }
  int version;
  *input >> version;
  cout << "reading binvox version " << version << endl;

  int depth, height, width;
  depth = -1;
  Vector norm_translate;
  Float norm_scale;
  
  int done = 0;
  while(input->good() && !done) {
    *input >> line;
    if (line.compare("data") == 0) done = 1;
    else if (line.compare("dim") == 0) {
      *input >> depth >> height >> width;
    }
    else if (line.compare("translate") == 0) {
      *input >> norm_translate[X] >> norm_translate[Y] >> norm_translate[Z];
    }
    else if (line.compare("scale") == 0) {
      *input >> norm_scale;
    }
    else {
      cout << "  unrecognized keyword [" << line << "], skipping" << endl;
      char c;
      do {
	c = input->get();
      } while(input->good() && (c != '\n'));
    }
  }
  if (!done) {
    cout << "  error reading header" << endl;
    return 0;
  }
  if (depth == -1) {
    cout << "  missing dimension in header" << endl;
    return 0;
  }
  
  voxels.init(width, height, depth, with_types);
  voxels.set_norm_translate(norm_translate);
  voxels.set_norm_scale(norm_scale);
  
  //
  // read voxel data
  //
  byte value;
  byte count;
  int index = 0;
  int end_index = 0;
  int nr_voxels = 0;
  
  input->unsetf(ios::skipws);  // !! need to read every byte now !!
  *input >> value;  // read the linefeed char
  int size = voxels.get_size();
  
  while((end_index < size) && input->good()) {
    *input >> value >> count;
    //    cout << "Read value: " << (int) value << ", count: " << (int) count << endl;

    if (input->good()) {
      end_index = index + count;
      if (end_index > size) return 0;
      for(int i=index; i < end_index; i++) voxels[i] = value;
      
      if (value) nr_voxels += count;
      index = end_index;
    }  // if file still ok
    
  }  // while

  input->close();
  cout << "  read " << nr_voxels << " voxels" << endl;
  voxels.init_types();
  voxels.update_voxel_refs();

  //
  // for debugging
  //
  //   ofstream vox_coords("vox_coords.dat", ios::out);
  //   for(int i=0; i < depth; i++) {
  //     for(int k=0; k < height; k++) {
  //       for(int j=0; j < width; j++) {
  // 	int index = voxels.get_index(i, j, k);
  // 	if (voxels[index]) {
  // 	  vox_coords << i << " " << j << " " << k << endl;
  // 	}
  //       }
  //     }
  //   }
  //   vox_coords.close();
  
  return 1;
  
}  // VoxelFile::read_binvox



int
VoxelFile::read_vt()
{
  //
  // read header
  //
  string line;
  *input >> line;  // #vt
  if (line.compare("#vt") != 0) {
    cout << "Error: first line reads [" << line << "] instead of '#voxel'" << endl;
    delete input;
    return 0;
  }

  int depth, height, width;
  *input >> depth >> height >> width;
  cout << "d x h x w = [" << depth << " x " << height << " x " << width << "]" << endl;
  voxels.init(width, height, depth, with_types);

  byte value;
  byte count;
  int index = 0;
  int end_index = 0;
  int nr_voxels = 0;
  
  input->unsetf(ios::skipws);  // !! need to read every byte now !!
  *input >> value;  // read the linefeed char
  int size = voxels.get_size();
  
  //
  // read voxel data
  //
  while((end_index < size) && input->good()) {
    *input >> value >> count;
    //    cout << "Read value: " << (int) value << ", count: " << (int) count << endl;

    if (input->good()) {
      end_index = index + count;
      if (end_index > size) return 0;
      for(int i=index; i < end_index; i++) voxels[i] = 1 - value;
      
      if (!value) nr_voxels += count;
      index = end_index;
    }  // if file still ok
    
  }  // while


  //
  // read type data
  //
  end_index = 0;
  index = 0;
  int nr_types = 0;
  
  while((end_index < size) && input->good()) {
    *input >> value >> count;
    //    cout << "Read value: " << (int) value << ", count: " << (int) count << endl;

    if (input->good()) {
      end_index = index + count;
      if (end_index > size) return 0;
      for(int i=index; i < end_index; i++) voxels.set_type(i, value);
      
      if (!value) nr_types += count;
      index = end_index;
    }  // if file still ok
    
  }  // while
  
  input->close();

  cout << "  read " << nr_voxels << " voxels, " << nr_types << " types" << endl;

  voxels.update_voxel_refs();
  
  return 1;
  
  
}  // VoxelFile::read_vt



int
VoxelFile::write_file()
{
  cout << "VoxelFile::write_file(" << total_filespec << ")" << endl;

  switch(my_filetype) {
    case BINVOX: return write_binvox();
    case VT: return write_vt();
    case MIRA: return write_mira();
    case RAWVOX: return write_raw_vox();
    case VTK: return write_vtk();
    default:
      cout << "  error: unknown filetype " << my_filetype << endl;
  }  // switch

  return 0;
  
}  // Voxels::write_file



int
VoxelFile::write_binvox()
{
  int width, height, depth;
  voxels.get_dimensions(&width, &height, &depth);

  Vector norm_translate = voxels.get_norm_translate();
  Float norm_scale = voxels.get_norm_scale();

  //
  // write header
  //
  *output << "#binvox 1" << endl;
  //  *output << "bbox [-1,1][-1,1][-1,1]" << endl;  // no use for 'bbox'
  //  *output << "dim [" << depth << "," << height << "," << width << "]" << endl;
  //  *output << "type RLE" << endl;
  *output << "dim " << depth << " " << height << " " << width << endl;
  *output << "translate " << norm_translate[X] << " " << norm_translate[Y] << " " << norm_translate[Z] << endl;
  *output << "scale " << norm_scale << endl;
  *output << "data" << endl;
  
  byte value;
  byte count;
  int index = 0;
  int bytes_written = 0;
  int size = voxels.get_size();
  int total_ones = 0;
  
  while (index < size) {

    value = voxels[index];
    count = 0;
    while((index < size) && (count < 255) && (value == voxels[index])) {
      index++;
      count++;
    }
    //    value = 1 - value;
    if (value) total_ones += count;
    
    *output << value << count;  // inverted...
    bytes_written += 2;
    
  }  // while

  output->close();

  cout << "Wrote " << total_ones << " set voxels out of " << size << ", in "
       << bytes_written << " bytes" << endl;

  return 1;

}  // VoxelFile::write_binvox



int
VoxelFile::write_vt()
{
  int width, height, depth;
  voxels.get_dimensions(&width, &height, &depth);
  
  //
  // write header
  //
  *output << "#vt" << endl;
  *output << depth << endl;
  *output << height << endl;
  *output << width << endl;
  
  byte value;
  byte count;
  int index = 0;
  int bytes_written = 0;
  int size = voxels.get_size();
  
  while (index < size) {

    value = voxels[index];
    count = 0;
    while((index < size) && (count < 255) && (value == voxels[index])) {
      index++;
      count++;
    }
    value = 1 - value;
    
    *output << value << count;  // inverted...
    bytes_written += 2;
    
  }  // while

  //
  // now write the types
  //
  index = 0;
  int type_bytes_written = 0;
  while (index < size) {
    
    value = voxels.get_type(index);
    count = 0;
    while((index < size) && (count < 255) && (value == voxels.get_type(index))) {
      index++;
      count++;
    }
    *output << value << count;
    type_bytes_written += 2;
    
  }  // while
  
  output->close();
  
  cout << "  wrote " << size << " voxels, in " << bytes_written << " bytes, types in "
       << type_bytes_written << " bytes" << endl;
  return 1;
  
}  // VoxelFile::write_vt



//
// from asc source code, file vortex.hh
//
// Multiresolution Isosurface Extraction with Adaptive Skeleton Climbing
//
// Tim Poston , Tien-Tsin Wong , and Pheng-Ann Heng,
// Computer Graphics Forum, Vol. 17, No. 3, September 1998, pp. 137-148. 
//
typedef struct  {
  char fileid[5];
  byte control_z;
  unsigned short version;
  unsigned short xres;
  unsigned short yres;
  unsigned short zres;
  unsigned short flags;
  unsigned int map_offset;
  unsigned int voxel_offset;
  byte unused[104];
  char text[128];
 } fileheader;

int
VoxelFile::write_mira()
{
  int w, h, d;
  voxels.get_dimensions(&w, &h, &d);

  unsigned short width = w;
  unsigned short height = h;
  unsigned short depth = d;

  fileheader fh;
  strcpy(fh.fileid, "VOXEL");
  fh.control_z = 0x1a;  // 26
  fh.version = 0;
  fh.xres = swap_byte_order(depth);
  fh.yres = swap_byte_order(width);
  fh.zres = swap_byte_order(height);
  fh.flags = 0;
  fh.map_offset = sizeof(fileheader);
  fh.voxel_offset = fh.map_offset + (depth + width + height) * sizeof(double);

  output->write((const char *) &fh, sizeof(fh));

  double *xmap = new double[depth];
  double *ymap = new double[width];
  double *zmap = new double[height];
  for(int i=0; i < depth; i++) xmap[i] = swap_byte_order(i * 0.01);
  for(int i=0; i < width; i++) ymap[i] = swap_byte_order(i * 0.01);
  for(int i=0; i < height; i++) zmap[i] = swap_byte_order(i * 0.01);

  output->write((const char *) xmap, depth * sizeof(double));
  output->write((const char *) ymap, width * sizeof(double));
  output->write((const char *) zmap, height * sizeof(double));

  int size = depth * height * width;
  for(int i=0; i < size; i++) voxels[i] *= 255;

  output->write((const char *) voxels.get_voxels_p(), depth * height * width);
  
  output->close();

  return 1;
  
}  // VoxelFile::write_mira



int
VoxelFile::read_mira()
{
  cout << "VoxelFile::read_mira" << endl;

  fileheader fh;
  input->read((char *) &fh, sizeof(fh));

  fh.control_z = 0;
  char *id_string = (char *) &fh;
  if (strcmp(id_string, "VOXEL") != 0) {
    cout << "  error: expected VOXEL at the start of the file" << endl;
    input->close();
    delete input;
    return 0;
  }

  cout << "  version: " << fh.version << endl;
  unsigned short depth = fh.xres;  // swap_byte_order(fh.xres)
  unsigned short width = fh.yres;
  unsigned short height = fh.zres;
  depth = swap_byte_order(depth), width = swap_byte_order(width), height = swap_byte_order(height);
  
  cout << "  dimensions (d x h x w): " << depth << " x " << height << " x " << width << endl;
  voxels.init(width, height, depth, with_types);

  // skip the map
  int map_size = fh.voxel_offset - fh.map_offset;
  byte *map_buffer = new byte[map_size];
  input->read((char *) map_buffer, map_size);
  
  // now read the voxels
  int size = depth * height * width;
  input->read((char *) voxels.get_voxels_p(), size);

  // set voxels to 1
  int nr_voxels = 0;
  for(int i=0; i < size; i++) {
    if (voxels[i]) nr_voxels++;
    if (voxels[i] == 255) voxels[i] = 1;
  }  // for

  cout << "  read " << nr_voxels << " voxels" << endl;
  voxels.init_types();
  voxels.update_voxel_refs();

  input->close();
  delete input;
  return 1;

}  // VoxelFile::read_mira



int
VoxelFile::write_raw_vox()
{
  cout << "VoxelFile::write_raw_vox" << endl;

  int width, height, depth;
  voxels.get_dimensions(&width, &height, &depth);
  int size = width * height * depth;

  byte *rv = new byte[size];
  if (!rv) {
    cout << "  error: could not allocate " << size << " bytes for raw voxels" << endl;
    return 0;
  }

  for(int i=0; i < size; i++) rv[i] = 255 * voxels[i];
  
  output->write((const char *) rv, depth * height * width);
  delete[] rv;
  
  output->close();
  return 1;
  
}  // VoxelFile::write_raw_vox



int
VoxelFile::write_vtk()
{
  cout << "VoxelFile::write_vtk" << endl;

  *output << "# vtk DataFile Version 3.0" << endl;
  *output << vtk_comment_string << endl;
  *output << "BINARY" << endl;
  *output << "DATASET STRUCTURED_POINTS" << endl;
  *output << "DIMENSIONS " << depth << " " << width << " " << height << endl;
  *output << "ORIGIN 0 0 0" << endl;
  *output << "SPACING 1 1 1" << endl;
  
  int size = depth * height * width;
  *output << "POINT_DATA " << size << endl;
  *output << "SCALARS voxel_data unsigned_char" << endl;
  *output << "LOOKUP_TABLE default" << endl;

  byte *rv = new byte[size];
  if (!rv) {
    cout << "  error: could not allocate " << size << " bytes for vtk voxels" << endl;
    return 0;
  }

  for(int i=0; i < size; i++) rv[i] = 128 * voxels[i];
  
  output->write((const char *) rv, depth * height * width);
  delete[] rv;

  output->close();

  return 1;
  
}  // VoxelFile::write_vtk



