#include <iostream>
#include <string>
#include <string.h>
#ifndef IRIX
#include <cstdlib>
#endif
#include <math.h>
#include <assert.h>
#include "cc/geom/MeshFileIdentifier.h"
#include "cc/voxel/Voxelizer.h"
#include "cc/voxel/VoxelFile.h"
#include "cc/time/WallTimer.h"
#include "cc/ui/GLwindow.h"

#include "version.h"

static string model_filespec = "";
static string model_prefix = "";
static string model_extension = "";
static string voxel_extension = "";
static int voxel_type;

static int carving_only			= 0;
static int dilated_carving		= 0;
static int dilate				= 0;
static int voting_only			= 0;
static int voxel_dim			= Voxelizer::DEFAULT_WIDTH;
static int use_fbo				= 0;
static Mesh* mesh_p;

// used to compute reconstruction transform from voxel coordinates to mesh coordinates
Vector	   bmin, bmax;
static int set_bounding_box = 0;	//whether the bounding box is auto-computed from the mesh or explicitly given by user
Vector	   norm_translate;
Float	   norm_scale;

static GLwindow *ortho_win;			//the window, onscreen or in a FBO, where the GL slice rendering takes place



using namespace std;





// idle callback for voxelization: does the whole job, once we have a mesh
//
void voxelize()
{
  //
  // 4th parameter 0 so no types array is allocated in Voxels
  //
  Voxels *voxels = new Voxels(voxel_dim, voxel_dim, voxel_dim, 0);
  if (!voxels) { cout << "Error: not enough memory for voxel array" << endl; return; }
  
  Voxelizer *voxelizer = new Voxelizer(*voxels, *mesh_p, ortho_win);
  voxelizer->set_dilated(dilated_carving);
  
  if (carving_only) voxelizer->carve_voxelize();
  else voxelizer->parity_vote_voxelize(voting_only);				// voting_only = 'skip_simple' parameter, so this function will not call carve_voxelize

  if (dilate)														// dilate if requested
    for(int i=0; i < dilate; i++) 
	{
      cout << "  dilation step " << i+1 << endl;
      voxels->dilate();
    }

  voxels->set_norm_translate(norm_translate);						// set mesh coord correspondence data
  voxels->set_norm_scale(norm_scale);
  
  cout << endl << "writing voxel file..." << endl;
  VoxelFile* voxel_file = new VoxelFile(*voxels, model_prefix);
  voxel_file->open_for_write(voxel_type);
  voxel_file->write_file();
  cout << "done" << endl << endl;
  
  exit(0);  
}





int parse_args(int argc, char **argv)
{
  argc--; argv++;

  while (argc > 0) 
  {
    if ((*argv)[0] == '-') {
      if (!strcmp(*argv, "-t")) {
	argv++; argc--;
	voxel_extension = *argv;
	voxel_type = VoxelFile::get_filetype(voxel_extension);
	if (voxel_type == -1) {
	  cout << "Error: unknown voxel file type [" << voxel_extension << "]" << endl;
	  return 0;
	}
      }
      else if (!strcmp(*argv, "-v")) {
	voting_only = 1;
      }
      else if (!strcmp(*argv, "-c")) {
	carving_only = 1;
      }
	  else if (!strcmp(*argv, "-fb")) {
	use_fbo = 1;
	  }  
      else if (!strcmp(*argv, "-dc")) {
	dilated_carving = 1;
      }
      else if (!strcmp(*argv, "-dilate")) {
	argv++, argc--;
	dilate = atoi(*argv);
	cout << "  will dilate voxel model " << dilate << " times" << endl;
      }
      else if (!strcmp(*argv, "-bb")) {
	argv++, argc--;
	bmin[X] = atof(*argv);
	argv++, argc--;
	bmin[Y] = atof(*argv);
	argv++, argc--;
	bmin[Z] = atof(*argv);
	argv++, argc--;
	bmax[X] = atof(*argv);
	argv++, argc--;
	bmax[Y] = atof(*argv);
	argv++, argc--;
	bmax[Z] = atof(*argv);
	cout << "  force bounding box to " << bmin << " - " << bmax << endl;
	set_bounding_box = 1;
      }
      else if (!strcmp(*argv, "-d")) {
	argv++; argc--;
	voxel_dim = atoi(*argv);
      }
      else {
	cout << "Error: invalid program argument: [" << *argv << "]" << endl;
	return 0;
      }
    }
    else {
      model_filespec = *argv;
    }
    argv++; argc--;
  }  // while

  if (!model_filespec.length()) {
    cout << "Error: missing model filename" << endl;
    return 0;
  }
  if (voting_only && carving_only) {
    cout << " Note: both carving and voting is the default" << endl;
  }
  return 1;

}  // parse_args



void split_model_filespec()
{
  model_prefix = model_filespec;
  int dot_pos = model_prefix.rfind('.');
  if (dot_pos != string::npos) {
    model_extension = model_prefix.substr(dot_pos + 1);
    model_prefix.erase(dot_pos);
  }
  else model_extension = "";
  
}



int load_model_file()						//Load a mesh from the model_filespec file
{
  split_model_filespec();

  if (model_extension.length() == 0) {
    cout << "load_model_file error: missing extension in [" <<  model_filespec << "]" << endl;
    return 0;
  }

  mesh_p = new Mesh();
  mesh_p->clear();
  mesh_p->init();
  mesh_p->set_name(model_prefix);
  
  MeshFileIdentifier meshfile_identifier(*mesh_p);
  MeshFile *meshfile_p = meshfile_identifier.create_mesh_file(model_filespec);
  int result = meshfile_p->load();
  if (!result) return 0;
  
  if (mesh_p->get_nr_faces() == 0) {
    cout << "Error: mesh has no polygons" << endl;
    return 0;
  }

  if (mesh_p)									// if we do have a mesh:	
  {
    if (mesh_p->compute_bounding_box())			// mesh bounding box computed OK:	
	{
      if (set_bounding_box)						// using a user-specified voxel-bounding-box
	  {
		mesh_p->set_bounding_box(bmin, bmax);	// pass the desired voxel-bounding-box to the mesh
      }

      norm_scale = 1.0 / mesh_p->normalize();	// compute the transform (scale,translate) between the normalized mesh and the voxel volume
      mesh_p->get_bounding_box(&bmin, &bmax);	//

      norm_translate[X] = -0.025 * norm_scale + bmin[X];
      norm_translate[Y] = -0.025 * norm_scale + bmin[Y];
      norm_translate[Z] = -0.025 * norm_scale + bmin[Z];
    }
    else 
	{
      cout << "Error: could not compute bounding box (coords out of range)" << endl;
      return 0;									// no vertices to compute bbox from
    }
    
  }

  return result;
} 




int main(int argc, char **argv)
{
  cout << endl;
  cout << "--- [binvox] mesh voxelizer, version " << binvox_version << ", build #" << binvox_build_number << " on " << binvox_date << endl;
  cout << "--- original by Patrick Min, enhanced by Alex Telea 2004-2010" << endl;
  cout << endl;

  int args_ok = parse_args(argc, argv);

  if (!args_ok) {
    cout << endl;
    cout << "Usage: binvox [-d <voxel dimension>] [-t <voxel file type>] [-c] [-v] <model filespec>" << endl;
    cout << "  -d: specify voxel grid size (default " << Voxelizer::DEFAULT_WIDTH
	 << ", max " << Voxelizer::MAX_WIDTH << ")" << endl;
    cout << "  -t: specify voxel file type (default binvox, also supported: mira, vtk, raw)" << endl;
    cout << "  -c: z-buffer based carving method only" << endl;
    cout << "  -v: z-buffer based parity voting method only (default is both -c and -v)" << endl;
	cout << "  -fb: use framebuffer object (that is, work without an onscreen window)" << endl;
    cout << "Additional parameters:" << endl;
    cout << "  -bb <minx> <miny> <minz> <maxx> <maxy> <maxz>: force a different bounding box" << endl;
    cout << "Supported 3D model file formats:" << endl;
    cout << "  OBJ" << endl;
    cout << "Example:" << endl;
    cout << "binvox -c -d 200 -t mira plane.obj" << endl;
    cout << endl;
  }

  if (!args_ok) return 1;
  
  if (voxel_dim > Voxelizer::MAX_WIDTH) {
    cout << "Error: max voxel grid size is " << Voxelizer::MAX_WIDTH << endl << endl;
    return 1;
  }
  
  glutInit(&argc, argv);

  //
  // load model file
  //
  cout << "loading model file..." << endl;
  if (!load_model_file()) 
  {
    cout << endl;
    return 1;
  }

  //
  // voxelize
  //
  cout << endl << "voxel model dimension: " << voxel_dim << endl;
  ortho_win = new GLwindow("binvox", 0, 0, voxel_dim, voxel_dim, use_fbo);			//ALEX: last param says we want a FBO window
  
  if (use_fbo)																		//ALEX: if we use FBO's I don't see a reason to do the main loop thing..
    voxelize();																		//REMARK: I'm not 100% sure this is always correct on all platforms etc
  else
  {
    glutIdleFunc(voxelize);
    glutMainLoop();	  // start glut main loop, so voxelize func can be called
  }
  
  return 0;
  
}  // main

